<?php
require_once '../includes/config.php';
require_once '../includes/functions.php';

if (!isModerator()) {
    header('Location: /login.php');
    exit;
}

$title = "Модерация товаров";
include '../templates/admin/header.php';
include '../templates/admin/sidebar.php';

$db = db();
$moderatorId = $_SESSION['user_id'];

// Обработка действий модерации
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    $productId = (int)($_POST['product_id'] ?? 0);
    $comment = sanitize($_POST['comment'] ?? '');
    
    if ($productId && in_array($action, ['approve', 'reject'])) {
        $db->begin_transaction();
        
        try {
            // Обновляем статус товара
            $status = $action === 'approve' ? 'active' : 'inactive';
            $stmt = $db->prepare("
                UPDATE products 
                SET status = ?, 
                    requires_moderation = 0,
                    moderator_id = ?,
                    moderated_at = NOW() 
                WHERE id = ?
            ");
            $stmt->bind_param("sii", $status, $moderatorId, $productId);
            $stmt->execute();
            
            // Логируем действие
            $stmt = $db->prepare("
                INSERT INTO moderation_logs (product_id, moderator_id, action, comment, created_at)
                VALUES (?, ?, ?, ?, NOW())
            ");
            $stmt->bind_param("iiss", $productId, $moderatorId, $action, $comment);
            $stmt->execute();
            
            // Отправляем email автору
            if ($action === 'approve') {
                $stmt = $db->prepare("
                    SELECT u.email, p.name 
                    FROM products p 
                    JOIN users u ON p.created_by = u.id 
                    WHERE p.id = ?
                ");
                $stmt->bind_param("i", $productId);
                $stmt->execute();
                $result = $stmt->get_result();
                
                if ($row = $result->fetch_assoc()) {
                    $subject = "Ваш товар одобрен: " . $row['name'];
                    $message = "
                        <h2>Поздравляем!</h2>
                        <p>Ваш товар <strong>{$row['name']}</strong> был проверен и одобрен модератором.</p>
                        <p>Теперь он доступен в каталоге магазина.</p>
                    ";
                    sendEmail($row['email'], $subject, $message);
                }
            }
            
            $db->commit();
            
            header("Location: moderation.php?success=1");
            exit;
            
        } catch (Exception $e) {
            $db->rollback();
            $error = "Ошибка при модерации: " . $e->getMessage();
        }
    }
}

// Получаем товары на модерацию
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$limit = 20;
$offset = ($page - 1) * $limit;

$result = $db->query("
    SELECT p.*, u.username as author_name, u.email as author_email,
           c.name as category_name
    FROM products p
    LEFT JOIN users u ON p.created_by = u.id
    LEFT JOIN categories c ON p.category_id = c.id
    WHERE p.requires_moderation = 1
    ORDER BY p.created_at ASC
    LIMIT $limit OFFSET $offset
");

$totalResult = $db->query("SELECT COUNT(*) as total FROM products WHERE requires_moderation = 1");
$totalRow = $totalResult->fetch_assoc();
$totalItems = $totalRow['total'];
$totalPages = ceil($totalItems / $limit);
?>

<div class="container-fluid">
    <h2>Модерация товаров</h2>
    
    <?php if (isset($_GET['success'])): ?>
        <div class="alert alert-success alert-dismissible fade show">
            Действие успешно выполнено!
            <button type="button" class="close" data-dismiss="alert">
                <span>&times;</span>
            </button>
        </div>
    <?php endif; ?>
    
    <?php if (isset($error)): ?>
        <div class="alert alert-danger"><?php echo $error; ?></div>
    <?php endif; ?>
    
    <div class="card">
        <div class="card-header">
            <span class="badge badge-warning"><?php echo $totalItems; ?></span> товаров ожидают модерации
        </div>
        
        <div class="card-body">
            <?php if ($result->num_rows > 0): ?>
                <div class="table-responsive">
                    <table class="table table-hover">
                        <thead>
                            <tr>
                                <th>ID</th>
                                <th>Товар</th>
                                <th>Автор</th>
                                <th>Категория</th>
                                <th>Цена</th>
                                <th>Дата добавления</th>
                                <th>Действия</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php while ($row = $result->fetch_assoc()): ?>
                            <tr>
                                <td><?php echo $row['id']; ?></td>
                                <td>
                                    <strong><?php echo htmlspecialchars($row['name']); ?></strong>
                                    <br>
                                    <small class="text-muted"><?php echo substr($row['description'], 0, 100); ?>...</small>
                                </td>
                                <td>
                                    <?php echo htmlspecialchars($row['author_name']); ?>
                                    <br>
                                    <small><?php echo htmlspecialchars($row['author_email']); ?></small>
                                </td>
                                <td><?php echo htmlspecialchars($row['category_name']); ?></td>
                                <td><?php echo number_format($row['price'], 2); ?> ₽</td>
                                <td><?php echo date('d.m.Y H:i', strtotime($row['created_at'])); ?></td>
                                <td>
                                    <div class="btn-group btn-group-sm">
                                        <button type="button" class="btn btn-info" 
                                                data-toggle="modal" 
                                                data-target="#previewModal"
                                                data-id="<?php echo $row['id']; ?>"
                                                data-name="<?php echo htmlspecialchars($row['name']); ?>"
                                                data-description="<?php echo htmlspecialchars($row['description']); ?>"
                                                data-image="<?php echo $row['image']; ?>"
                                                data-price="<?php echo $row['price']; ?>">
                                            <i class="fas fa-eye"></i>
                                        </button>
                                        <button type="button" class="btn btn-success" 
                                                data-toggle="modal" 
                                                data-target="#approveModal"
                                                data-id="<?php echo $row['id']; ?>">
                                            <i class="fas fa-check"></i>
                                        </button>
                                        <button type="button" class="btn btn-danger"
                                                data-toggle="modal"
                                                data-target="#rejectModal"
                                                data-id="<?php echo $row['id']; ?>">
                                            <i class="fas fa-times"></i>
                                        </button>
                                    </div>
                                </td>
                            </tr>
                            <?php endwhile; ?>
                        </tbody>
                    </table>
                </div>
                
                <!-- Пагинация -->
                <?php if ($totalPages > 1): ?>
                <nav>
                    <ul class="pagination justify-content-center">
                        <?php for ($i = 1; $i <= $totalPages; $i++): ?>
                            <li class="page-item <?php echo $i === $page ? 'active' : ''; ?>">
                                <a class="page-link" href="?page=<?php echo $i; ?>">
                                    <?php echo $i; ?>
                                </a>
                            </li>
                        <?php endfor; ?>
                    </ul>
                </nav>
                <?php endif; ?>
                
            <?php else: ?>
                <div class="alert alert-info">
                    <i class="fas fa-check-circle mr-2"></i>
                    Все товары проверены! Ожидайте новых поступлений.
                </div>
            <?php endif; ?>
        </div>
    </div>
</div>

<!-- Модальное окно предпросмотра -->
<div class="modal fade" id="previewModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Предпросмотр товара</h5>
                <button type="button" class="close" data-dismiss="modal">
                    <span>&times;</span>
                </button>
            </div>
            <div class="modal-body">
                <div id="previewContent"></div>
            </div>
        </div>
    </div>
</div>

<!-- Модальное окно одобрения -->
<div class="modal fade" id="approveModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <form method="POST">
                <div class="modal-header">
                    <h5 class="modal-title">Одобрить товар</h5>
                    <button type="button" class="close" data-dismiss="modal">
                        <span>&times;</span>
                    </button>
                </div>
                <div class="modal-body">
                    <input type="hidden" name="action" value="approve">
                    <input type="hidden" name="product_id" id="approveProductId">
                    
                    <div class="form-group">
                        <label for="approveComment">Комментарий (необязательно)</label>
                        <textarea class="form-control" id="approveComment" name="comment" rows="3" 
                                  placeholder="Дополнительные заметки..."></textarea>
                    </div>
                    
                    <div class="alert alert-success">
                        <i class="fas fa-info-circle mr-2"></i>
                        Товар будет опубликован в каталоге после одобрения.
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-dismiss="modal">Отмена</button>
                    <button type="submit" class="btn btn-success">Одобрить</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Модальное окно отклонения -->
<div class="modal fade" id="rejectModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <form method="POST">
                <div class="modal-header">
                    <h5 class="modal-title">Отклонить товар</h5>
                    <button type="button" class="close" data-dismiss="modal">
                        <span>&times;</span>
                    </button>
                </div>
                <div class="modal-body">
                    <input type="hidden" name="action" value="reject">
                    <input type="hidden" name="product_id" id="rejectProductId">
                    
                    <div class="form-group">
                        <label for="rejectComment">Причина отклонения *</label>
                        <textarea class="form-control" id="rejectComment" name="comment" rows="3" 
                                  placeholder="Укажите причину отклонения товара..." required></textarea>
                        <small class="form-text text-muted">
                            Этот комментарий будет отправлен автору товара.
                        </small>
                    </div>
                    
                    <div class="alert alert-warning">
                        <i class="fas fa-exclamation-triangle mr-2"></i>
                        Товар будет скрыт из каталога. Автор получит уведомление.
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-dismiss="modal">Отмена</button>
                    <button type="submit" class="btn btn-danger">Отклонить</button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
// Обработка модальных окон
$('#previewModal').on('show.bs.modal', function(event) {
    const button = $(event.relatedTarget);
    const modal = $(this);
    
    const content = `
        <div class="row">
            <div class="col-md-4">
                <img src="${button.data('image') || '/assets/images/default-product.jpg'}" 
                     class="img-fluid rounded" alt="${button.data('name')}">
            </div>
            <div class="col-md-8">
                <h4>${button.data('name')}</h4>
                <p class="text-muted">ID: ${button.data('id')}</p>
                <p><strong>Цена:</strong> ${button.data('price')} ₽</p>
                <hr>
                <p>${button.data('description').replace(/\n/g, '<br>')}</p>
            </div>
        </div>
    `;
    
    modal.find('#previewContent').html(content);
});

$('#approveModal').on('show.bs.modal', function(event) {
    const button = $(event.relatedTarget);
    $(this).find('#approveProductId').val(button.data('id'));
});

$('#rejectModal').on('show.bs.modal', function(event) {
    const button = $(event.relatedTarget);
    $(this).find('#rejectProductId').val(button.data('id'));
});
</script>

<?php include '../templates/admin/footer.php'; ?>