<?php
require_once '../includes/config.php';
require_once '../includes/functions.php';
require_once '../includes/security.php';

header('Content-Type: application/json');

// Проверка авторизации и прав
if (!isAdmin()) {
    http_response_code(403);
    echo json_encode(['error' => 'Доступ запрещен']);
    exit;
}

// Проверка CSRF токена
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $csrfToken = $_SERVER['HTTP_X_CSRF_TOKEN'] ?? $_POST['csrf_token'] ?? '';
    
    if (!Security::validateCSRFToken($csrfToken)) {
        http_response_code(403);
        echo json_encode(['error' => 'Неверный CSRF токен']);
        exit;
    }
}

// Получение данных
$input = json_decode(file_get_contents('php://input'), true) ?: $_POST;
$action = $input['action'] ?? '';

$db = db();
$response = [];

switch ($action) {
    case 'delete_product':
        $productId = (int)($input['product_id'] ?? 0);
        
        if ($productId) {
            // Вместо удаления помечаем как неактивный
            $stmt = $db->prepare("UPDATE products SET status = 'inactive', updated_at = NOW() WHERE id = ?");
            $stmt->bind_param("i", $productId);
            
            if ($stmt->execute()) {
                $response = ['success' => true, 'message' => 'Товар деактивирован'];
            } else {
                $response = ['error' => 'Ошибка базы данных'];
            }
        } else {
            $response = ['error' => 'Не указан ID товара'];
        }
        break;
        
    case 'update_product_status':
        $productId = (int)($input['product_id'] ?? 0);
        $status = $input['status'] ?? '';
        
        if ($productId && in_array($status, ['active', 'inactive', 'draft'])) {
            $stmt = $db->prepare("UPDATE products SET status = ?, updated_at = NOW() WHERE id = ?");
            $stmt->bind_param("si", $status, $productId);
            
            if ($stmt->execute()) {
                $response = ['success' => true];
            } else {
                $response = ['error' => 'Ошибка обновления'];
            }
        }
        break;
        
    case 'get_statistics':
        // Статистика для дашборда
        $stats = [];
        
        // Общая статистика
        $result = $db->query("
            SELECT 
                (SELECT COUNT(*) FROM users) as total_users,
                (SELECT COUNT(*) FROM products WHERE status = 'active') as total_products,
                (SELECT COUNT(*) FROM orders WHERE payment_status = 'paid') as total_orders,
                (SELECT SUM(total_amount) FROM orders WHERE payment_status = 'paid') as total_revenue
        ");
        
        if ($row = $result->fetch_assoc()) {
            $stats['general'] = $row;
        }
        
        // Продажи за последние 30 дней
        $result = $db->query("
            SELECT 
                DATE(created_at) as date,
                COUNT(*) as orders_count,
                SUM(total_amount) as daily_revenue
            FROM orders 
            WHERE payment_status = 'paid' 
            AND created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)
            GROUP BY DATE(created_at)
            ORDER BY date DESC
        ");
        
        $stats['sales_chart'] = [];
        while ($row = $result->fetch_assoc()) {
            $stats['sales_chart'][] = $row;
        }
        
        // Популярные товары
        $result = $db->query("
            SELECT 
                p.name,
                p.price,
                COUNT(oi.product_id) as sales_count,
                SUM(oi.price) as revenue
            FROM order_items oi
            JOIN products p ON oi.product_id = p.id
            JOIN orders o ON oi.order_id = o.id
            WHERE o.payment_status = 'paid'
            GROUP BY oi.product_id
            ORDER BY sales_count DESC
            LIMIT 10
        ");
        
        $stats['top_products'] = [];
        while ($row = $result->fetch_assoc()) {
            $stats['top_products'][] = $row;
        }
        
        $response = ['success' => true, 'data' => $stats];
        break;
        
    case 'get_user_list':
        $page = (int)($input['page'] ?? 1);
        $limit = (int)($input['limit'] ?? 20);
        $offset = ($page - 1) * $limit;
        
        $result = $db->query("
            SELECT 
                u.*,
                (SELECT COUNT(*) FROM orders WHERE user_id = u.id AND payment_status = 'paid') as orders_count,
                (SELECT SUM(total_amount) FROM orders WHERE user_id = u.id AND payment_status = 'paid') as total_spent
            FROM users u
            ORDER BY u.created_at DESC
            LIMIT $limit OFFSET $offset
        ");
        
        $users = [];
        while ($row = $result->fetch_assoc()) {
            unset($row['password']);
            $users[] = $row;
        }
        
        // Общее количество
        $totalResult = $db->query("SELECT COUNT(*) as total FROM users");
        $total = $totalResult->fetch_assoc()['total'];
        
        $response = [
            'success' => true,
            'data' => $users,
            'pagination' => [
                'page' => $page,
                'limit' => $limit,
                'total' => $total,
                'pages' => ceil($total / $limit)
            ]
        ];
        break;
        
    case 'update_user_role':
        $userId = (int)($input['user_id'] ?? 0);
        $role = $input['role'] ?? '';
        
        if ($userId && in_array($role, ['user', 'moderator', 'admin'])) {
            // Не позволяем изменять свою роль
            if ($userId == $_SESSION['user_id']) {
                $response = ['error' => 'Нельзя изменить свою роль'];
                break;
            }
            
            $stmt = $db->prepare("UPDATE users SET role = ? WHERE id = ?");
            $stmt->bind_param("si", $role, $userId);
            
            if ($stmt->execute()) {
                $response = ['success' => true];
            } else {
                $response = ['error' => 'Ошибка обновления'];
            }
        }
        break;
        
    default:
        $response = ['error' => 'Неизвестное действие'];
        break;
}

echo json_encode($response);
?>