// Анимация появления элементов
function animateOnScroll() {
    const elements = document.querySelectorAll('.animate-on-scroll');
    
    if (elements.length === 0) return;
    
    const observer = new IntersectionObserver((entries) => {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                entry.target.classList.add('animated');
                observer.unobserve(entry.target);
            }
        });
    }, {
        threshold: 0.1,
        rootMargin: '0px 0px -50px 0px'
    });
    
    elements.forEach(el => observer.observe(el));
}

// Плавная прокрутка
function initSmoothScroll() {
    document.querySelectorAll('a[href^="#"]').forEach(anchor => {
        anchor.addEventListener('click', function(e) {
            const href = this.getAttribute('href');
            if (href === '#' || href === '#!') return;
            
            const targetElement = document.querySelector(href);
            if (targetElement) {
                e.preventDefault();
                
                // Прокручиваем к элементу
                const headerHeight = document.querySelector('.navbar')?.offsetHeight || 80;
                const targetPosition = targetElement.getBoundingClientRect().top + window.pageYOffset - headerHeight;
                
                window.scrollTo({
                    top: targetPosition,
                    behavior: 'smooth'
                });
                
                // Обновляем URL без перезагрузки
                if (history.pushState) {
                    history.pushState(null, null, href);
                }
            }
        });
    });
}

// Анимация добавления в корзину
function animateCartAdd(itemElement) {
    const cartIcon = document.getElementById('cart-icon');
    if (!cartIcon || !itemElement) return;
    
    const itemRect = itemElement.getBoundingClientRect();
    const cartRect = cartIcon.getBoundingClientRect();
    
    // Создаем летающий элемент
    const flyingItem = document.createElement('div');
    flyingItem.className = 'flying-item';
    flyingItem.style.cssText = `
        position: fixed;
        left: ${itemRect.left + itemRect.width / 2}px;
        top: ${itemRect.top + itemRect.height / 2}px;
        width: 20px;
        height: 20px;
        background: var(--primary-color, #3498db);
        border-radius: 50%;
        pointer-events: none;
        z-index: 9999;
        transition: all 0.6s cubic-bezier(0.175, 0.885, 0.32, 1.275);
    `;
    
    document.body.appendChild(flyingItem);
    
    // Запускаем анимацию
    requestAnimationFrame(() => {
        flyingItem.style.left = `${cartRect.left + cartRect.width / 2}px`;
        flyingItem.style.top = `${cartRect.top + cartRect.height / 2}px`;
        flyingItem.style.transform = 'scale(0.5)';
        flyingItem.style.opacity = '0.5';
    });
    
    // Удаляем после анимации
    setTimeout(() => {
        if (flyingItem.parentNode) {
            flyingItem.remove();
        }
        
        // Анимация иконки корзины
        cartIcon.classList.add('bounce');
        setTimeout(() => {
            cartIcon.classList.remove('bounce');
        }, 300);
    }, 600);
}

// Инициализация анимаций при загрузке
document.addEventListener('DOMContentLoaded', function() {
    // Добавляем классы для анимации
    document.querySelectorAll('.product-card, .category-card, .step-card').forEach(card => {
        card.classList.add('animate-on-scroll');
    });
    
    // Инициализация
    animateOnScroll();
    initSmoothScroll();
    
    // Обработка кнопок "В корзину"
    document.addEventListener('click', function(e) {
        if (e.target.closest('.add-to-cart')) {
            const button = e.target.closest('.add-to-cart');
            const productCard = button.closest('.product-card, .card');
            
            if (productCard) {
                animateCartAdd(productCard);
            }
        }
    });
    
    // Обработка изменения темы для анимаций
    const themeObserver = new MutationObserver((mutations) => {
        mutations.forEach(mutation => {
            if (mutation.attributeName === 'data-theme') {
                // Переинициализация анимаций при смене темы
                setTimeout(animateOnScroll, 100);
            }
        });
    });
    
    themeObserver.observe(document.documentElement, {
        attributes: true,
        attributeFilter: ['data-theme']
    });
});

// Анимация счетчика
function animateCounter(element, targetValue, duration = 1000) {
    if (!element) return;
    
    const startValue = parseInt(element.textContent) || 0;
    const increment = targetValue > startValue ? 1 : -1;
    const stepTime = Math.abs(Math.floor(duration / (targetValue - startValue)));
    
    let currentValue = startValue;
    const timer = setInterval(() => {
        currentValue += increment;
        element.textContent = currentValue;
        
        if (currentValue === targetValue) {
            clearInterval(timer);
        }
    }, stepTime);
}

// Анимация появления уведомлений
function animateNotification(element) {
    if (!element) return;
    
    element.style.opacity = '0';
    element.style.transform = 'translateY(-20px)';
    
    requestAnimationFrame(() => {
        element.style.transition = 'opacity 0.3s, transform 0.3s';
        element.style.opacity = '1';
        element.style.transform = 'translateY(0)';
    });
}

// Анимация загрузки
function showLoadingAnimation(element) {
    if (!element) return;
    
    const originalContent = element.innerHTML;
    element.innerHTML = '<span class="loading"></span>';
    element.disabled = true;
    
    return function() {
        element.innerHTML = originalContent;
        element.disabled = false;
    };
}

// Экспорт функций для использования в других скриптах
if (typeof window !== 'undefined') {
    window.AppAnimations = {
        animateOnScroll,
        initSmoothScroll,
        animateCartAdd,
        animateCounter,
        animateNotification,
        showLoadingAnimation
    };
}