// Ленивая загрузка изображений
class LazyLoader {
    constructor(options = {}) {
        this.options = {
            rootMargin: '50px 0px',
            threshold: 0.01,
            ...options
        };
        
        this.observer = null;
        this.images = [];
        this.init();
    }
    
    init() {
        if ('IntersectionObserver' in window) {
            this.observer = new IntersectionObserver((entries, observer) => {
                entries.forEach(entry => {
                    if (entry.isIntersecting) {
                        this.loadImage(entry.target);
                        observer.unobserve(entry.target);
                    }
                });
            }, this.options);
            
            this.findImages();
        } else {
            // Fallback для старых браузеров
            this.loadAllImages();
        }
    }
    
    findImages() {
        this.images = Array.from(document.querySelectorAll('img[data-src], img[data-srcset]'));
        this.images.forEach(img => {
            // Добавляем placeholder если нужно
            if (!img.src && !img.hasAttribute('src')) {
                img.src = 'data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iMzAwIiBoZWlnaHQ9IjIwMCIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj48cmVjdCB3aWR0aD0iMTAwJSIgaGVpZ2h0PSIxMDAlIiBmaWxsPSIjZjJmNGY3Ii8+PC9zdmc+';
            }
            
            this.observer.observe(img);
        });
    }
    
    loadImage(img) {
        const src = img.getAttribute('data-src');
        const srcset = img.getAttribute('data-srcset');
        
        if (src) {
            img.src = src;
            img.removeAttribute('data-src');
        }
        
        if (srcset) {
            img.srcset = srcset;
            img.removeAttribute('data-srcset');
        }
        
        // Добавляем класс для CSS анимации
        img.classList.add('lazy-loaded');
        
        // Событие после загрузки
        img.addEventListener('load', () => {
            img.classList.add('lazy-load-complete');
        });
    }
    
    loadAllImages() {
        this.images.forEach(img => this.loadImage(img));
    }
    
    addImage(img) {
        if (this.observer) {
            this.observer.observe(img);
        } else {
            this.loadImage(img);
        }
    }
    
    destroy() {
        if (this.observer) {
            this.observer.disconnect();
        }
    }
}

// Инициализация при загрузке страницы
document.addEventListener('DOMContentLoaded', function() {
    window.lazyLoader = new LazyLoader();
    
    // Обработка динамически добавленных изображений
    const observer = new MutationObserver((mutations) => {
        mutations.forEach(mutation => {
            if (mutation.addedNodes.length) {
                mutation.addedNodes.forEach(node => {
                    if (node.nodeType === 1) { // Element node
                        const images = node.querySelectorAll ? node.querySelectorAll('img[data-src]') : [];
                        images.forEach(img => window.lazyLoader.addImage(img));
                        
                        if (node.tagName === 'IMG' && node.hasAttribute('data-src')) {
                            window.lazyLoader.addImage(node);
                        }
                    }
                });
            }
        });
    });
    
    observer.observe(document.body, {
        childList: true,
        subtree: true
    });
});

// Функция для принудительной загрузки изображения
function forceLoadImage(img) {
    if (window.lazyLoader) {
        window.lazyLoader.loadImage(img);
    }
}

// Экспорт для использования в других скриптах
if (typeof window !== 'undefined') {
    window.LazyLoader = LazyLoader;
    window.forceLoadImage = forceLoadImage;
}