<?php
require_once 'includes/config.php';
require_once 'includes/functions.php';

$title = "Корзина покупок";
$theme = isset($_COOKIE['theme']) ? $_COOKIE['theme'] : 'light';

// Получаем товары из корзины
$cart_items = [];
$total = 0;
$discount = 0;
$final_total = 0;

if (isLoggedIn()) {
    $db = db();
    $user_id = $_SESSION['user_id'];
    
    $stmt = $db->prepare("
        SELECT c.id as cart_id, c.quantity, c.added_at, 
               p.id, p.name, p.price, p.discount_price, p.image, p.description,
               (SELECT name FROM categories WHERE id = p.category_id) as category_name
        FROM cart c
        JOIN products p ON c.product_id = p.id
        WHERE c.user_id = ? AND p.status = 'active'
        ORDER BY c.added_at DESC
    ");
    $stmt->bind_param("i", $user_id);
    $stmt->execute();
    $result = $stmt->get_result();
    
    while ($item = $result->fetch_assoc()) {
        $price = $item['discount_price'] ?: $item['price'];
        $item_total = $price * $item['quantity'];
        
        $cart_items[] = [
            'cart_id' => $item['cart_id'],
            'product_id' => $item['id'],
            'name' => $item['name'],
            'price' => $price,
            'original_price' => $item['price'],
            'quantity' => $item['quantity'],
            'image' => $item['image'],
            'description' => $item['description'],
            'category' => $item['category_name'],
            'item_total' => $item_total
        ];
        
        $total += $item['price'] * $item['quantity'];
        $final_total += $item_total;
    }
    
    $discount = $total - $final_total;
} elseif (isset($_SESSION['cart']) && is_array($_SESSION['cart'])) {
    // Для неавторизованных пользователей
    $db = db();
    $cart_ids = array_keys($_SESSION['cart']);
    
    if (!empty($cart_ids)) {
        $placeholders = str_repeat('?,', count($cart_ids) - 1) . '?';
        $stmt = $db->prepare("
            SELECT p.id, p.name, p.price, p.discount_price, p.image, p.description,
                   (SELECT name FROM categories WHERE id = p.category_id) as category_name
            FROM products p
            WHERE p.id IN ($placeholders) AND p.status = 'active'
        ");
        
        $types = str_repeat('i', count($cart_ids));
        $stmt->bind_param($types, ...$cart_ids);
        $stmt->execute();
        $result = $stmt->get_result();
        
        while ($item = $result->fetch_assoc()) {
            $quantity = $_SESSION['cart'][$item['id']];
            $price = $item['discount_price'] ?: $item['price'];
            $item_total = $price * $quantity;
            
            $cart_items[] = [
                'product_id' => $item['id'],
                'name' => $item['name'],
                'price' => $price,
                'original_price' => $item['price'],
                'quantity' => $quantity,
                'image' => $item['image'],
                'description' => $item['description'],
                'category' => $item['category_name'],
                'item_total' => $item_total
            ];
            
            $total += $item['price'] * $quantity;
            $final_total += $item_total;
        }
        
        $discount = $total - $final_total;
    }
}

// Обработка действий с корзиной
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    $product_id = isset($_POST['product_id']) ? (int)$_POST['product_id'] : 0;
    $quantity = isset($_POST['quantity']) ? (int)$_POST['quantity'] : 1;
    
    if ($action === 'update') {
        // Обновление количества
        if ($quantity > 0) {
            if (isLoggedIn()) {
                $db = db();
                $stmt = $db->prepare("UPDATE cart SET quantity = ? WHERE user_id = ? AND product_id = ?");
                $stmt->bind_param("iii", $quantity, $_SESSION['user_id'], $product_id);
                $stmt->execute();
            } else {
                $_SESSION['cart'][$product_id] = $quantity;
            }
            
            $_SESSION['notification'] = [
                'type' => 'success',
                'message' => 'Количество товара обновлено'
            ];
        }
    } elseif ($action === 'remove') {
        // Удаление товара
        if (isLoggedIn()) {
            $db = db();
            $stmt = $db->prepare("DELETE FROM cart WHERE user_id = ? AND product_id = ?");
            $stmt->bind_param("ii", $_SESSION['user_id'], $product_id);
            $stmt->execute();
        } else {
            unset($_SESSION['cart'][$product_id]);
        }
        
        $_SESSION['notification'] = [
            'type' => 'success',
            'message' => 'Товар удален из корзины'
        ];
    } elseif ($action === 'clear') {
        // Очистка корзины
        if (isLoggedIn()) {
            $db = db();
            $stmt = $db->prepare("DELETE FROM cart WHERE user_id = ?");
            $stmt->bind_param("i", $_SESSION['user_id']);
            $stmt->execute();
        } else {
            $_SESSION['cart'] = [];
        }
        
        $_SESSION['notification'] = [
            'type' => 'success',
            'message' => 'Корзина очищена'
        ];
    }
    
    header("Location: /cart.php");
    exit;
}

include 'templates/header.php';
?>

<div class="container py-5">
    <h1 class="mb-4"><i class="fas fa-shopping-cart me-2"></i>Корзина покупок</h1>
    
    <?php if (empty($cart_items)): ?>
        <div class="card">
            <div class="card-body text-center py-5">
                <i class="fas fa-shopping-cart fa-4x text-muted mb-4"></i>
                <h3 class="text-muted mb-3">Ваша корзина пуста</h3>
                <p class="text-muted mb-4">Добавьте товары из каталога, чтобы оформить заказ</p>
                <a href="/catalog.php" class="btn btn-primary btn-lg">
                    <i class="fas fa-shopping-bag me-2"></i>Перейти в каталог
                </a>
            </div>
        </div>
    <?php else: ?>
        <div class="row">
            <!-- Товары в корзине -->
            <div class="col-lg-8">
                <div class="card mb-4">
                    <div class="card-header d-flex justify-content-between align-items-center">
                        <h5 class="mb-0">Товары в корзине (<?php echo count($cart_items); ?>)</h5>
                        <form method="POST" onsubmit="return confirm('Вы уверены, что хотите очистить корзину?');">
                            <input type="hidden" name="action" value="clear">
                            <button type="submit" class="btn btn-sm btn-outline-danger">
                                <i class="fas fa-trash me-1"></i>Очистить корзину
                            </button>
                        </form>
                    </div>
                    
                    <div class="card-body p-0">
                        <div class="table-responsive">
                            <table class="table table-hover mb-0">
                                <thead class="table-light">
                                    <tr>
                                        <th style="width: 120px;">Товар</th>
                                        <th>Название</th>
                                        <th style="width: 100px;">Цена</th>
                                        <th style="width: 140px;">Количество</th>
                                        <th style="width: 100px;">Сумма</th>
                                        <th style="width: 60px;"></th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($cart_items as $item): ?>
                                    <tr>
                                        <td>
                                            <?php if ($item['image']): ?>
                                                <img src="<?php echo $item['image']; ?>" 
                                                     alt="<?php echo htmlspecialchars($item['name']); ?>" 
                                                     class="img-fluid rounded" 
                                                     style="width: 80px; height: 80px; object-fit: cover;">
                                            <?php else: ?>
                                                <div class="bg-light rounded d-flex align-items-center justify-content-center" 
                                                     style="width: 80px; height: 80px;">
                                                    <i class="fas fa-box text-muted fa-2x"></i>
                                                </div>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <h6 class="mb-1">
                                                <a href="/product.php?id=<?php echo $item['product_id']; ?>" 
                                                   class="text-decoration-none">
                                                    <?php echo htmlspecialchars($item['name']); ?>
                                                </a>
                                            </h6>
                                            <p class="text-muted small mb-0">
                                                <i class="fas fa-tag me-1"></i>
                                                <?php echo htmlspecialchars($item['category']); ?>
                                            </p>
                                            <?php if ($item['price'] < $item['original_price']): ?>
                                                <span class="badge bg-success small">
                                                    <i class="fas fa-percent me-1"></i>
                                                    Скидка <?php echo round(($item['original_price'] - $item['price']) / $item['original_price'] * 100); ?>%
                                                </span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <div class="price">
                                                <?php if ($item['price'] < $item['original_price']): ?>
                                                    <div class="text-muted small text-decoration-line-through">
                                                        <?php echo number_format($item['original_price'], 2); ?> ₽
                                                    </div>
                                                <?php endif; ?>
                                                <div class="fw-bold">
                                                    <?php echo number_format($item['price'], 2); ?> ₽
                                                </div>
                                            </div>
                                        </td>
                                        <td>
                                            <form method="POST" class="quantity-form">
                                                <input type="hidden" name="action" value="update">
                                                <input type="hidden" name="product_id" value="<?php echo $item['product_id']; ?>">
                                                <div class="input-group input-group-sm">
                                                    <button class="btn btn-outline-secondary minus-btn" type="button">-</button>
                                                    <input type="number" 
                                                           name="quantity" 
                                                           value="<?php echo $item['quantity']; ?>" 
                                                           min="1" 
                                                           max="99"
                                                           class="form-control text-center quantity-input">
                                                    <button class="btn btn-outline-secondary plus-btn" type="button">+</button>
                                                </div>
                                                <button type="submit" class="btn btn-link btn-sm mt-1 text-decoration-none">
                                                    <i class="fas fa-sync-alt me-1"></i>Обновить
                                                </button>
                                            </form>
                                        </td>
                                        <td>
                                            <div class="fw-bold h6 mb-0">
                                                <?php echo number_format($item['item_total'], 2); ?> ₽
                                            </div>
                                        </td>
                                        <td>
                                            <form method="POST" onsubmit="return confirm('Удалить товар из корзины?');">
                                                <input type="hidden" name="action" value="remove">
                                                <input type="hidden" name="product_id" value="<?php echo $item['product_id']; ?>">
                                                <button type="submit" class="btn btn-link text-danger" title="Удалить">
                                                    <i class="fas fa-times"></i>
                                                </button>
                                            </form>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
                
                <!-- Продолжить покупки -->
                <div class="d-flex justify-content-between">
                    <a href="/catalog.php" class="btn btn-outline-primary">
                        <i class="fas fa-arrow-left me-2"></i>Продолжить покупки
                    </a>
                    
                    <div class="dropdown">
                        <button class="btn btn-outline-secondary dropdown-toggle" type="button" data-bs-toggle="dropdown">
                            <i class="fas fa-share-alt me-2"></i>Поделиться корзиной
                        </button>
                        <ul class="dropdown-menu">
                            <li><a class="dropdown-item" href="#" onclick="shareCart('whatsapp')"><i class="fab fa-whatsapp me-2"></i>WhatsApp</a></li>
                            <li><a class="dropdown-item" href="#" onclick="shareCart('telegram')"><i class="fab fa-telegram me-2"></i>Telegram</a></li>
                            <li><a class="dropdown-item" href="#" onclick="copyCartLink()"><i class="fas fa-link me-2"></i>Скопировать ссылку</a></li>
                        </ul>
                    </div>
                </div>
            </div>
            
            <!-- Итоги и оформление -->
            <div class="col-lg-4">
                <div class="card sticky-top" style="top: 20px;">
                    <div class="card-header">
                        <h5 class="mb-0">Итоги заказа</h5>
                    </div>
                    
                    <div class="card-body">
                        <div class="mb-3">
                            <div class="d-flex justify-content-between mb-2">
                                <span>Товаров на сумму:</span>
                                <span><?php echo number_format($total, 2); ?> ₽</span>
                            </div>
                            
                            <?php if ($discount > 0): ?>
                            <div class="d-flex justify-content-between mb-2 text-success">
                                <span>Скидка:</span>
                                <span>-<?php echo number_format($discount, 2); ?> ₽</span>
                            </div>
                            <?php endif; ?>
                            
                            <div class="d-flex justify-content-between mb-2">
                                <span>Доставка:</span>
                                <span class="text-success">Бесплатно</span>
                            </div>
                            
                            <hr>
                            
                            <div class="d-flex justify-content-between mb-3">
                                <strong>Итого к оплате:</strong>
                                <strong class="h5 text-primary"><?php echo number_format($final_total, 2); ?> ₽</strong>
                            </div>
                        </div>
                        
                        <?php if (isLoggedIn()): ?>
                            <a href="/checkout.php" class="btn btn-primary btn-lg w-100 mb-3">
                                <i class="fas fa-credit-card me-2"></i>Перейти к оформлению
                            </a>
                        <?php else: ?>
                            <div class="alert alert-warning">
                                <i class="fas fa-exclamation-triangle me-2"></i>
                                Для оформления заказа необходимо 
                                <a href="/login.php?redirect=/cart.php" class="alert-link">войти</a> 
                                или <a href="/register.php" class="alert-link">зарегистрироваться</a>
                            </div>
                            
                            <div class="d-grid gap-2">
                                <a href="/login.php?redirect=/cart.php" class="btn btn-primary">
                                    <i class="fas fa-sign-in-alt me-2"></i>Войти
                                </a>
                                <a href="/register.php" class="btn btn-outline-primary">
                                    <i class="fas fa-user-plus me-2"></i>Регистрация
                                </a>
                            </div>
                        <?php endif; ?>
                        
                        <hr class="my-3">
                        
                        <div class="small text-muted">
                            <p class="mb-2">
                                <i class="fas fa-shield-alt me-2"></i>
                                Безопасная оплата через ЮMoney и банковские карты
                            </p>
                            <p class="mb-2">
                                <i class="fas fa-download me-2"></i>
                                Мгновенная доставка после оплаты
                            </p>
                            <p class="mb-0">
                                <i class="fas fa-headset me-2"></i>
                                Поддержка 24/7
                            </p>
                        </div>
                    </div>
                </div>
                
                <!-- Промокод -->
                <div class="card mt-3">
                    <div class="card-body">
                        <h6 class="card-title"><i class="fas fa-tag me-2"></i>Промокод</h6>
                        <form id="promoForm">
                            <div class="input-group">
                                <input type="text" class="form-control" placeholder="Введите промокод" id="promoCode">
                                <button class="btn btn-outline-primary" type="submit">Применить</button>
                            </div>
                            <div id="promoMessage" class="mt-2 small"></div>
                        </form>
                    </div>
                </div>
            </div>
        </div>
    <?php endif; ?>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    // Управление количеством товаров
    document.querySelectorAll('.minus-btn').forEach(btn => {
        btn.addEventListener('click', function() {
            const input = this.closest('.input-group').querySelector('.quantity-input');
            const value = parseInt(input.value) || 1;
            if (value > 1) {
                input.value = value - 1;
            }
        });
    });
    
    document.querySelectorAll('.plus-btn').forEach(btn => {
        btn.addEventListener('click', function() {
            const input = this.closest('.input-group').querySelector('.quantity-input');
            const value = parseInt(input.value) || 1;
            if (value < 99) {
                input.value = value + 1;
            }
        });
    });
    
    // Обновление количества при изменении
    document.querySelectorAll('.quantity-input').forEach(input => {
        input.addEventListener('change', function() {
            if (this.value < 1) this.value = 1;
            if (this.value > 99) this.value = 99;
        });
    });
    
    // Автоотправка формы при обновлении количества
    document.querySelectorAll('.quantity-form').forEach(form => {
        form.addEventListener('submit', function(e) {
            e.preventDefault();
            
            const formData = new FormData(this);
            const productId = formData.get('product_id');
            const quantity = formData.get('quantity');
            
            fetch('/api/cart.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-Token': window.App.csrfToken
                },
                body: JSON.stringify({
                    action: 'update',
                    product_id: productId,
                    quantity: quantity
                })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    location.reload();
                } else {
                    alert('Ошибка: ' + data.error);
                }
            });
        });
    });
    
    // Промокод
    const promoForm = document.getElementById('promoForm');
    if (promoForm) {
        promoForm.addEventListener('submit', function(e) {
            e.preventDefault();
            const promoCode = document.getElementById('promoCode').value;
            const message = document.getElementById('promoMessage');
            
            if (!promoCode.trim()) {
                message.innerHTML = '<span class="text-danger">Введите промокод</span>';
                return;
            }
            
            fetch('/api/apply-promo.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-Token': window.App.csrfToken
                },
                body: JSON.stringify({ promo_code: promoCode })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    message.innerHTML = '<span class="text-success">' + data.message + '</span>';
                    if (data.discount) {
                        // Обновляем сумму на странице
                        setTimeout(() => location.reload(), 1000);
                    }
                } else {
                    message.innerHTML = '<span class="text-danger">' + data.error + '</span>';
                }
            });
        });
    }
});

function shareCart(platform) {
    const cartItems = <?php echo json_encode(array_column($cart_items, 'name')); ?>;
    const total = <?php echo $final_total; ?>;
    const itemsCount = <?php echo count($cart_items); ?>;
    
    let text = `Моя корзина покупок:\n`;
    text += `Товаров: ${itemsCount} на сумму ${total} ₽\n\n`;
    
    cartItems.forEach((item, index) => {
        text += `${index + 1}. ${item}\n`;
    });
    
    text += `\nСсылка на магазин: ${window.location.origin}`;
    
    let url = '';
    switch(platform) {
        case 'whatsapp':
            url = `https://wa.me/?text=${encodeURIComponent(text)}`;
            break;
        case 'telegram':
            url = `https://t.me/share/url?url=${encodeURIComponent(window.location.origin)}&text=${encodeURIComponent(text)}`;
            break;
    }
    
    if (url) {
        window.open(url, '_blank');
    }
}

function copyCartLink() {
    const link = window.location.href;
    navigator.clipboard.writeText(link).then(() => {
        showNotification('Ссылка на корзину скопирована в буфер обмена', 'success');
    });
}
</script>

<?php include 'templates/footer.php'; ?>