<?php
require_once 'includes/config.php';
require_once 'includes/functions.php';

$title = "Товары со скидками";
$theme = isset($_COOKIE['theme']) ? $_COOKIE['theme'] : 'light';

$db = db();

// Получаем товары со скидками
$page = isset($_GET['page']) ? max(1, (int)$_GET['page']) : 1;
$limit = 12;
$offset = ($page - 1) * $limit;

$products = $db->query("
    SELECT SQL_CALC_FOUND_ROWS 
        p.*, 
        c.name as category_name,
        c.slug as category_slug,
        p.discount_price,
        p.price,
        ROUND((p.price - p.discount_price) / p.price * 100) as discount_percent
    FROM products p
    LEFT JOIN categories c ON p.category_id = c.id
    WHERE p.status = 'active' 
    AND p.discount_price IS NOT NULL
    AND p.discount_price < p.price
    ORDER BY discount_percent DESC, p.created_at DESC
    LIMIT $limit OFFSET $offset
");

// Общее количество
$total_result = $db->query("SELECT FOUND_ROWS() as total");
$total_row = $total_result->fetch_assoc();
$total_products = $total_row['total'];
$total_pages = ceil($total_products / $limit);

// Получаем товары с максимальной скидкой
$max_discounts = $db->query("
    SELECT p.*, c.name as category_name,
           ROUND((p.price - p.discount_price) / p.price * 100) as discount_percent
    FROM products p
    LEFT JOIN categories c ON p.category_id = c.id
    WHERE p.status = 'active' 
    AND p.discount_price IS NOT NULL
    AND p.discount_price < p.price
    ORDER BY discount_percent DESC
    LIMIT 3
");

// Получаем общую сумму сэкономленных денег
$total_savings = $db->query("
    SELECT SUM(p.price - p.discount_price) as total_savings
    FROM products p
    WHERE p.status = 'active' 
    AND p.discount_price IS NOT NULL
    AND p.discount_price < p.price
")->fetch_assoc()['total_savings'] ?? 0;

include 'templates/header.php';
?>

<div class="container py-5">
    <!-- Заголовок с описанием -->
    <div class="row mb-5">
        <div class="col-12">
            <h1 class="display-5 mb-3">
                <i class="fas fa-percent text-success me-2"></i>Товары со скидками
            </h1>
            <p class="lead text-muted">
                Лучшие предложения и специальные цены. 
                Покупайте цифровые товары по выгодной стоимости и экономьте свои деньги!
            </p>
            
            <!-- Статистика скидок -->
            <div class="row mt-4">
                <div class="col-md-4 mb-3">
                    <div class="card border-success">
                        <div class="card-body text-center">
                            <h2 class="text-success mb-0"><?php echo $total_products; ?></h2>
                            <p class="text-muted mb-0">товаров со скидкой</p>
                        </div>
                    </div>
                </div>
                <div class="col-md-4 mb-3">
                    <div class="card border-warning">
                        <div class="card-body text-center">
                            <h2 class="text-warning mb-0">до 90%</h2>
                            <p class="text-muted mb-0">максимальная скидка</p>
                        </div>
                    </div>
                </div>
                <div class="col-md-4 mb-3">
                    <div class="card border-primary">
                        <div class="card-body text-center">
                            <h2 class="text-primary mb-0"><?php echo number_format($total_savings, 0); ?> ₽</h2>
                            <p class="text-muted mb-0">можно сэкономить</p>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Товары с максимальной скидкой -->
    <?php if ($max_discounts->num_rows > 0): ?>
    <div class="row mb-5">
        <div class="col-12">
            <h3 class="mb-4">
                <i class="fas fa-crown text-warning me-2"></i>Самые выгодные предложения
            </h3>
            
            <div class="row">
                <?php 
                while ($product = $max_discounts->fetch_assoc()): 
                    $discount_percent = round(($product['price'] - $product['discount_price']) / $product['price'] * 100);
                    $savings = $product['price'] - $product['discount_price'];
                ?>
                <div class="col-md-4 mb-4">
                    <div class="card h-100 border-warning shadow-lg">
                        <!-- Большой бейдж скидки -->
                        <div class="position-absolute top-0 start-0 m-3">
                            <div class="discount-badge bg-danger text-white rounded-circle d-flex flex-column align-items-center justify-content-center" 
                                 style="width: 80px; height: 80px;">
                                <span class="fs-4 fw-bold">-<?php echo $discount_percent; ?>%</span>
                                <small>СКИДКА</small>
                            </div>
                        </div>
                        
                        <!-- Изображение товара -->
                        <div class="card-img-container" style="height: 200px;">
                            <img src="<?php echo $product['image'] ?: '/assets/images/default-product.jpg'; ?>" 
                                 class="card-img-top h-100" 
                                 alt="<?php echo htmlspecialchars($product['name']); ?>"
                                 style="object-fit: cover;">
                        </div>
                        
                        <div class="card-body">
                            <h5 class="card-title">
                                <a href="/product.php?id=<?php echo $product['id']; ?>" class="text-decoration-none">
                                    <?php echo htmlspecialchars($product['name']); ?>
                                </a>
                            </h5>
                            
                            <p class="card-text text-muted small mb-3">
                                <i class="fas fa-tag me-1"></i>
                                <?php echo htmlspecialchars($product['category_name']); ?>
                            </p>
                            
                            <div class="price mb-3">
                                <div class="text-muted text-decoration-line-through mb-1">
                                    <?php echo number_format($product['price'], 2); ?> ₽
                                </div>
                                <div class="h3 text-success mb-0">
                                    <?php echo number_format($product['discount_price'], 2); ?> ₽
                                </div>
                                <div class="text-success small">
                                    <i class="fas fa-piggy-bank me-1"></i>
                                    Экономия <?php echo number_format($savings, 2); ?> ₽
                                </div>
                            </div>
                            
                            <div class="d-grid">
                                <button class="btn btn-warning btn-lg add-to-cart" data-product-id="<?php echo $product['id']; ?>">
                                    <i class="fas fa-cart-plus me-2"></i>Купить со скидкой
                                </button>
                            </div>
                        </div>
                        
                        <div class="card-footer bg-transparent border-top-0">
                            <div class="d-flex justify-content-between small text-muted">
                                <span>
                                    <i class="fas fa-download me-1"></i>
                                    <?php echo number_format($product['downloads_count']); ?>
                                </span>
                                <span>
                                    <i class="fas fa-clock me-1"></i>
                                    <?php echo date('d.m.Y', strtotime($product['created_at'])); ?>
                                </span>
                            </div>
                        </div>
                    </div>
                </div>
                <?php endwhile; ?>
            </div>
        </div>
    </div>
    <?php endif; ?>
    
    <!-- Все товары со скидками -->
    <div class="row mb-4">
        <div class="col-12">
            <h3 class="border-bottom pb-3">
                <i class="fas fa-tags text-success me-2"></i>Все товары со скидками
                <span class="badge bg-success ms-2"><?php echo $total_products; ?> товаров</span>
            </h3>
            <p class="text-muted">
                Специальные предложения и акционные товары. Скидки действуют ограниченное время, успейте приобрести!
            </p>
        </div>
    </div>
    
    <?php if ($products->num_rows > 0): ?>
        <div class="row row-cols-1 row-cols-md-2 row-cols-lg-3 g-4 mb-5">
            <?php while ($product = $products->fetch_assoc()): 
                $discount_percent = $product['discount_percent'];
                $savings = $product['price'] - $product['discount_price'];
            ?>
            <div class="col">
                <div class="card h-100 product-card position-relative">
                    <!-- Бейдж скидки -->
                    <div class="position-absolute top-0 start-0 m-2">
                        <span class="badge bg-danger fs-6">
                            -<?php echo $discount_percent; ?>%
                        </span>
                    </div>
                    
                    <!-- Бейдж "Выгода" -->
                    <div class="position-absolute top-0 end-0 m-2">
                        <span class="badge bg-success">
                            <i class="fas fa-piggy-bank me-1"></i>Экономия <?php echo number_format($savings, 0); ?> ₽
                        </span>
                    </div>
                    
                    <div class="card-img-container">
                        <img src="<?php echo $product['image'] ?: '/assets/images/default-product.jpg'; ?>" 
                             class="card-img-top" 
                             alt="<?php echo htmlspecialchars($product['name']); ?>"
                             loading="lazy">
                    </div>
                    
                    <div class="card-body">
                        <h5 class="card-title">
                            <a href="/product.php?id=<?php echo $product['id']; ?>" class="text-decoration-none">
                                <?php echo htmlspecialchars($product['name']); ?>
                            </a>
                        </h5>
                        
                        <p class="card-text text-muted small">
                            <i class="fas fa-tag me-1"></i>
                            <a href="/catalog.php?category=<?php echo $product['category_id']; ?>" class="text-decoration-none text-muted">
                                <?php echo htmlspecialchars($product['category_name']); ?>
                            </a>
                        </p>
                        
                        <p class="card-text small">
                            <?php echo htmlspecialchars(mb_substr($product['description'], 0, 100)) . (mb_strlen($product['description']) > 100 ? '...' : ''); ?>
                        </p>
                        
                        <div class="price mt-3">
                            <div class="text-muted text-decoration-line-through small">
                                <?php echo number_format($product['price'], 2); ?> ₽
                            </div>
                            <div class="h4 text-success mb-0">
                                <?php echo number_format($product['discount_price'], 2); ?> ₽
                            </div>
                        </div>
                        
                        <div class="d-flex justify-content-between align-items-center mt-3">
                            <div class="text-success small">
                                <i class="fas fa-bolt me-1"></i>
                                Скидка <?php echo $discount_percent; ?>%
                            </div>
                            
                            <div class="btn-group">
                                <a href="/product.php?id=<?php echo $product['id']; ?>" class="btn btn-sm btn-outline-success">
                                    Подробнее
                                </a>
                                <button class="btn btn-sm btn-success add-to-cart" data-product-id="<?php echo $product['id']; ?>">
                                    <i class="fas fa-cart-plus"></i>
                                </button>
                            </div>
                        </div>
                    </div>
                    
                    <div class="card-footer bg-transparent">
                        <div class="progress mb-2" style="height: 5px;">
                            <div class="progress-bar bg-warning" 
                                 style="width: <?php echo min($discount_percent, 100); ?>%">
                            </div>
                        </div>
                        <div class="d-flex justify-content-between small text-muted">
                            <span>
                                <i class="fas fa-download me-1"></i>
                                <?php echo number_format($product['downloads_count']); ?>
                            </span>
                            <span>
                                <i class="fas fa-calendar me-1"></i>
                                <?php echo date('d.m.Y', strtotime($product['created_at'])); ?>
                            </span>
                        </div>
                    </div>
                </div>
            </div>
            <?php endwhile; ?>
        </div>
        
        <!-- Пагинация -->
        <?php if ($total_pages > 1): ?>
        <nav aria-label="Навигация по страницам">
            <ul class="pagination justify-content-center">
                <?php if ($page > 1): ?>
                <li class="page-item">
                    <a class="page-link" href="?page=<?php echo $page - 1; ?>">
                        <i class="fas fa-chevron-left"></i>
                    </a>
                </li>
                <?php endif; ?>
                
                <?php
                $start = max(1, $page - 2);
                $end = min($total_pages, $page + 2);
                
                if ($start > 1) {
                    echo '<li class="page-item"><a class="page-link" href="?page=1">1</a></li>';
                    if ($start > 2) {
                        echo '<li class="page-item disabled"><span class="page-link">...</span></li>';
                    }
                }
                
                for ($i = $start; $i <= $end; $i++):
                ?>
                <li class="page-item <?php echo $i == $page ? 'active' : ''; ?>">
                    <a class="page-link" href="?page=<?php echo $i; ?>">
                        <?php echo $i; ?>
                    </a>
                </li>
                <?php endfor; ?>
                
                if ($end < $total_pages) {
                    if ($end < $total_pages - 1) {
                        echo '<li class="page-item disabled"><span class="page-link">...</span></li>';
                    }
                    echo '<li class="page-item"><a class="page-link" href="?page=' . $total_pages . '">' . $total_pages . '</a></li>';
                }
                ?>
                
                <?php if ($page < $total_pages): ?>
                <li class="page-item">
                    <a class="page-link" href="?page=<?php echo $page + 1; ?>">
                        <i class="fas fa-chevron-right"></i>
                    </a>
                </li>
                <?php endif; ?>
            </ul>
        </nav>
        <?php endif; ?>
        
    <?php else: ?>
        <div class="card">
            <div class="card-body text-center py-5">
                <i class="fas fa-percent fa-4x text-muted mb-4"></i>
                <h3 class="text-muted mb-3">Товары со скидками не найдены</h3>
                <p class="text-muted mb-4">
                    В настоящее время акционных товаров нет. Загляните позже или посмотрите другие разделы.
                </p>
                <div class="d-grid gap-2 d-md-block">
                    <a href="/catalog.php" class="btn btn-primary me-md-2">
                        <i class="fas fa-shopping-bag me-2"></i>Весь каталог
                    </a>
                    <a href="/new.php" class="btn btn-outline-primary me-md-2">
                        <i class="fas fa-bolt me-2"></i>Новинки
                    </a>
                    <a href="/popular.php" class="btn btn-outline-success">
                        <i class="fas fa-fire me-2"></i>Популярное
                    </a>
                </div>
            </div>
        </div>
    <?php endif; ?>
    
    <!-- Акции и промокоды -->
    <div class="row mt-5">
        <div class="col-12">
            <div class="card bg-gradient-success text-white">
                <div class="card-body">
                    <div class="row align-items-center">
                        <div class="col-md-8">
                            <h4 class="card-title mb-3">
                                <i class="fas fa-gift me-2"></i>Дополнительные скидки
                            </h4>
                            <p class="card-text mb-0">
                                Используйте промокоды для получения дополнительных скидок на уже сниженные товары!
                            </p>
                        </div>
                        <div class="col-md-4 text-md-end">
                            <div class="input-group">
                                <input type="text" class="form-control" placeholder="Введите промокод" id="promoCodeInput">
                                <button class="btn btn-light" type="button" id="applyPromoBtn">
                                    <i class="fas fa-check"></i>
                                </button>
                            </div>
                            <div id="promoMessage" class="mt-2 small"></div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Почему выгодно покупать со скидкой -->
    <div class="row mt-5">
        <div class="col-12">
            <div class="card">
                <div class="card-body">
                    <h5 class="card-title mb-4">
                        <i class="fas fa-lightbulb text-warning me-2"></i>Почему стоит покупать товары со скидкой?
                    </h5>
                    
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <div class="d-flex">
                                <div class="flex-shrink-0">
                                    <i class="fas fa-money-bill-wave text-success fa-2x me-3"></i>
                                </div>
                                <div class="flex-grow-1">
                                    <h6>Реальная экономия</h6>
                                    <p class="small text-muted mb-0">
                                        Покупайте те же качественные товары, но по более низкой цене. 
                                        Скидки позволяют сэкономить до 90% от первоначальной стоимости.
                                    </p>
                                </div>
                            </div>
                        </div>
                        
                        <div class="col-md-6 mb-3">
                            <div class="d-flex">
                                <div class="flex-shrink-0">
                                    <i class="fas fa-award text-primary fa-2x me-3"></i>
                                </div>
                                <div class="flex-grow-1">
                                    <h6>Проверенное качество</h6>
                                    <p class="small text-muted mb-0">
                                        Товары со скидкой проходят такую же проверку качества, как и обычные товары. 
                                        Скидка не означает ухудшение качества.
                                    </p>
                                </div>
                            </div>
                        </div>
                        
                        <div class="col-md-6 mb-3">
                            <div class="d-flex">
                                <div class="flex-shrink-0">
                                    <i class="fas fa-clock text-danger fa-2x me-3"></i>
                                </div>
                                <div class="flex-grow-1">
                                    <h6>Ограниченное время</h6>
                                    <p class="small text-muted mb-0">
                                        Скидки действуют ограниченное время. Успейте приобрести товары по специальной цене, 
                                        пока предложение действительно.
                                    </p>
                                </div>
                            </div>
                        </div>
                        
                        <div class="col-md-6 mb-3">
                            <div class="d-flex">
                                <div class="flex-shrink-0">
                                    <i class="fas fa-sync-alt text-info fa-2x me-3"></i>
                                </div>
                                <div class="flex-grow-1">
                                    <h6>Обновление ассортимента</h6>
                                    <p class="small text-muted mb-0">
                                        Мы регулярно обновляем список товаров со скидками. 
                                        Подпишитесь на уведомления, чтобы не пропустить новые выгодные предложения.
                                    </p>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<style>
.bg-gradient-success {
    background: linear-gradient(135deg, #11998e 0%, #38ef7d 100%) !important;
}

.discount-badge {
    box-shadow: 0 4px 10px rgba(220, 53, 69, 0.4);
    z-index: 1;
}

.product-card .badge.bg-danger {
    font-size: 1rem;
    padding: 0.5rem 0.75rem;
}
</style>

<script>
document.addEventListener('DOMContentLoaded', function() {
    // Применение промокода
    const applyPromoBtn = document.getElementById('applyPromoBtn');
    const promoCodeInput = document.getElementById('promoCodeInput');
    const promoMessage = document.getElementById('promoMessage');
    
    if (applyPromoBtn && promoCodeInput) {
        applyPromoBtn.addEventListener('click', function() {
            const promoCode = promoCodeInput.value.trim();
            
            if (!promoCode) {
                promoMessage.innerHTML = '<span class="text-warning">Введите промокод</span>';
                return;
            }
            
            fetch('/api/apply-promo.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-Token': window.App.csrfToken
                },
                body: JSON.stringify({ promo_code: promoCode, page: 'discounts' })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    promoMessage.innerHTML = '<span class="text-white">' + data.message + '</span>';
                    if (data.discount) {
                        // Сохраняем промокод в localStorage для использования при оформлении заказа
                        localStorage.setItem('active_promo_code', promoCode);
                        localStorage.setItem('promo_discount', data.discount);
                    }
                } else {
                    promoMessage.innerHTML = '<span class="text-warning">' + data.error + '</span>';
                }
            });
        });
        
        // Применение по Enter
        promoCodeInput.addEventListener('keypress', function(e) {
            if (e.key === 'Enter') {
                applyPromoBtn.click();
            }
        });
    }
    
    // Анимация для бейджей скидок
    document.querySelectorAll('.discount-badge').forEach(badge => {
        badge.addEventListener('mouseenter', function() {
            this.style.transform = 'scale(1.1) rotate(5deg)';
            this.style.transition = 'transform 0.3s';
        });
        
        badge.addEventListener('mouseleave', function() {
            this.style.transform = 'scale(1) rotate(0)';
        });
    });
});
</script>

<?php include 'templates/footer.php'; ?>