<?php
error_reporting(E_ALL);
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);

require_once 'includes/config.php';
require_once 'includes/functions.php';
require_once 'includes/auth.php';
require_once 'includes/security.php';

// Если пользователь уже авторизован, перенаправляем
if (isLoggedIn()) {
    header('Location: /profile.php');
    exit;
}

$error = '';
$success = '';

// Обработка формы входа
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $email = sanitize($_POST['email'] ?? '');
    $password = $_POST['password'] ?? '';
    $remember = isset($_POST['remember']);
    
    // Проверка CSRF токена
    if (!isset($_POST['csrf_token']) || $_POST['csrf_token'] !== $_SESSION['csrf_token']) {
        $error = 'Неверный CSRF токен';
    } else {
        // Проверка попыток входа
        if (!Security::checkLoginAttempts($_SERVER['REMOTE_ADDR'])) {
            $error = 'Слишком много попыток входа. Попробуйте через 30 минут.';
        } else {
            $result = loginUser($email, $password);
            
            if (isset($result['success'])) {
                // Запоминаем пользователя если выбрано "Запомнить меня"
                if ($remember) {
                    $token = bin2hex(random_bytes(32));
                    $expires = date('Y-m-d H:i:s', time() + 60*60*24*30); // 30 дней
                    
                    $db = db();
                    $stmt = $db->prepare("
                        INSERT INTO remember_tokens (user_id, token, expires_at, created_at) 
                        VALUES (?, ?, ?, NOW())
                        ON DUPLICATE KEY UPDATE 
                        token = VALUES(token), 
                        expires_at = VALUES(expires_at)
                    ");
                    $stmt->bind_param("iss", $_SESSION['user_id'], $token, $expires);
                    $stmt->execute();
                    
                    setcookie('remember_token', $token, time() + 60*60*24*30, '/', '', true, true);
                }
                
                // Перенаправление
                $redirect = $_GET['redirect'] ?? '/profile.php';
                header("Location: $redirect");
                exit;
            } else {
                $error = $result['error'];
                Security::recordLoginAttempt($_SERVER['REMOTE_ADDR'], false);
            }
        }
    }
}

// Генерация CSRF токена
if (empty($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

$title = "Вход в аккаунт";
$theme = isset($_COOKIE['theme']) ? $_COOKIE['theme'] : 'light';

include 'templates/header.php';
?>

<div class="container py-5">
    <div class="row justify-content-center">
        <div class="col-md-6 col-lg-5">
            <div class="card shadow">
                <div class="card-header bg-primary text-white">
                    <h4 class="mb-0"><i class="fas fa-sign-in-alt me-2"></i>Вход в аккаунт</h4>
                </div>
                
                <div class="card-body p-4">
                    <?php if ($error): ?>
                        <div class="alert alert-danger alert-dismissible fade show">
                            <?php echo $error; ?>
                            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                        </div>
                    <?php endif; ?>
                    
                    <?php if ($success): ?>
                        <div class="alert alert-success alert-dismissible fade show">
                            <?php echo $success; ?>
                            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                        </div>
                    <?php endif; ?>
                    
                    <form method="POST" id="loginForm">
                        <input type="hidden" name="csrf_token" value="<?php echo $_SESSION['csrf_token']; ?>">
                        
                        <div class="mb-3">
                            <label for="email" class="form-label">
                                <i class="fas fa-envelope me-1"></i>Email адрес
                            </label>
                            <input type="email" 
                                   class="form-control <?php echo isset($error) && strpos($error, 'email') !== false ? 'is-invalid' : ''; ?>" 
                                   id="email" 
                                   name="email" 
                                   value="<?php echo htmlspecialchars($_POST['email'] ?? ''); ?>" 
                                   required
                                   autocomplete="email"
                                   autofocus>
                            <div class="invalid-feedback">
                                Пожалуйста, введите корректный email
                            </div>
                        </div>
                        
                        <div class="mb-3">
                            <label for="password" class="form-label">
                                <i class="fas fa-lock me-1"></i>Пароль
                            </label>
                            <div class="input-group">
                                <input type="password" 
                                       class="form-control <?php echo isset($error) && strpos($error, 'пароль') !== false ? 'is-invalid' : ''; ?>" 
                                       id="password" 
                                       name="password" 
                                       required
                                       autocomplete="current-password">
                                <button class="btn btn-outline-secondary" type="button" id="togglePassword">
                                    <i class="fas fa-eye"></i>
                                </button>
                            </div>
                            <div class="invalid-feedback">
                                Неверный пароль
                            </div>
                            <div class="mt-1">
                                <a href="/forgot-password.php" class="small text-decoration-none">Забыли пароль?</a>
                            </div>
                        </div>
                        
                        <div class="mb-3 form-check">
                            <input type="checkbox" class="form-check-input" id="remember" name="remember" checked>
                            <label class="form-check-label" for="remember">Запомнить меня</label>
                        </div>
                        
                        <div class="d-grid gap-2">
                            <button type="submit" class="btn btn-primary btn-lg">
                                <i class="fas fa-sign-in-alt me-2"></i>Войти
                            </button>
                            
                            <div class="text-center my-3">
                                <span class="text-muted">или</span>
                            </div>
                            
                            <a href="/register.php" class="btn btn-outline-primary">
                                <i class="fas fa-user-plus me-2"></i>Создать аккаунт
                            </a>
                        </div>
                    </form>
                </div>
                
                <div class="card-footer text-center bg-light">
                    <p class="mb-0 small text-muted">
                        Входя в аккаунт, вы соглашаетесь с 
                        <a href="/terms.php" class="text-decoration-none">пользовательским соглашением</a> 
                        и <a href="/privacy.php" class="text-decoration-none">политикой конфиденциальности</a>
                    </p>
                </div>
            </div>
            
            <!-- Информация для новых пользователей -->
            <div class="card mt-4">
                <div class="card-body">
                    <h6 class="card-title"><i class="fas fa-info-circle me-2"></i>Почему стоит зарегистрироваться?</h6>
                    <ul class="list-unstyled small">
                        <li class="mb-2"><i class="fas fa-check text-success me-2"></i> История заказов</li>
                        <li class="mb-2"><i class="fas fa-check text-success me-2"></i> Быстрое оформление покупок</li>
                        <li class="mb-2"><i class="fas fa-check text-success me-2"></i> Скидки для постоянных клиентов</li>
                        <li class="mb-2"><i class="fas fa-check text-success me-2"></i> Отслеживание статуса заказов</li>
                        <li><i class="fas fa-check text-success me-2"></i> Личный кабинет</li>
                    </ul>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    // Переключение видимости пароля
    const togglePassword = document.getElementById('togglePassword');
    const passwordInput = document.getElementById('password');
    
    if (togglePassword && passwordInput) {
        togglePassword.addEventListener('click', function() {
            const type = passwordInput.getAttribute('type') === 'password' ? 'text' : 'password';
            passwordInput.setAttribute('type', type);
            this.innerHTML = type === 'password' ? '<i class="fas fa-eye"></i>' : '<i class="fas fa-eye-slash"></i>';
        });
    }
    
    // Валидация формы
    const loginForm = document.getElementById('loginForm');
    if (loginForm) {
        loginForm.addEventListener('submit', function(e) {
            const email = document.getElementById('email').value;
            const password = document.getElementById('password').value;
            
            let isValid = true;
            
            // Проверка email
            const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
            if (!emailRegex.test(email)) {
                document.getElementById('email').classList.add('is-invalid');
                isValid = false;
            }
            
            // Проверка пароля
            if (password.length < 6) {
                document.getElementById('password').classList.add('is-invalid');
                isValid = false;
            }
            
            if (!isValid) {
                e.preventDefault();
            }
        });
    }
});
</script>

<?php include 'templates/footer.php'; ?>