<?php

require_once __DIR__ . '/../config/bootstrap.php';
require_once __DIR__ . '/../config/auth.php';
require_once __DIR__ . '/../config/functions.php';

if (!is_admin()) {
    http_response_code(403);
    die("Доступ запрещён");
}

$user_id = (int)($_GET['id'] ?? 0);
if ($user_id <= 0) {
    die("Неверный ID пользователя");
}

try {

    $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?");
    $stmt->execute([$user_id]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$user) {
        die("Пользователь не найден");
    }
} catch (Exception $e) {
    die("Ошибка загрузки данных пользователя: " . $e->getMessage());
}

require_once __DIR__ . '/../includes/header.php';
?>

<div class="container my-5">
    <h1 class="mb-4">
        <i class="bi bi-person-gear"></i> Редактирование пользователя
    </h1>
    
    <div class="row">
        <div class="col-md-6">
            <div class="glass p-4 card-ui fade-in">
                <form id="edit-user-form">
                    <?= csrf_input(); ?>
                    <input type="hidden" name="id" value="<?= e($user['id']) ?>">
                    
                    <div class="mb-3">
                        <label for="email" class="form-label fw-bold">
                            <i class="bi bi-envelope"></i> Email *
                        </label>
                        <input type="email" class="form-control" id="email" name="email" 
                               required value="<?= e($user['email']) ?>">
                        <div class="invalid-feedback">Введите корректный email</div>
                    </div>
                    
                    <div class="mb-3">
                        <label for="name" class="form-label fw-bold">
                            <i class="bi bi-person"></i> Имя *
                        </label>
                        <input type="text" class="form-control" id="name" name="name" 
                               required value="<?= e($user['name']) ?>">
                        <div class="invalid-feedback">Введите имя пользователя</div>
                    </div>
                    
                    <div class="mb-3">
                        <label for="phone" class="form-label fw-bold">
                            <i class="bi bi-telephone"></i> Телефон
                        </label>
                        <input type="tel" class="form-control" id="phone" name="phone" 
                               value="<?= e($user['phone'] ?? '') ?>"
                               placeholder="+7 (999) 123-45-67">
                    </div>
                    
                    <div class="mb-3">
                        <label for="role" class="form-label fw-bold">
                            <i class="bi bi-shield"></i> Роль *
                        </label>
                        <select class="form-select" id="role" name="role" required>
                            <option value="user" <?= $user['role'] === 'user' ? 'selected' : '' ?>>Пользователь</option>
                            <option value="admin" <?= $user['role'] === 'admin' ? 'selected' : '' ?>>Администратор</option>
                        </select>
                    </div>
                    
                    <div class="mb-3">
                        <label for="notifications" class="form-label fw-bold">
                            <i class="bi bi-bell"></i> Уведомления
                        </label>
                        <select class="form-select" id="notifications" name="notifications">
                            <option value="1" <?= $user['notifications'] ? 'selected' : '' ?>>Включены</option>
                            <option value="0" <?= !$user['notifications'] ? 'selected' : '' ?>>Выключены</option>
                        </select>
                    </div>
                    
                    <div class="mb-4">
                        <label for="password" class="form-label fw-bold">
                            <i class="bi bi-key"></i> Новый пароль
                        </label>
                        <input type="password" class="form-control" id="password" name="password" 
                               minlength="6" placeholder="Оставьте пустым, если не меняете">
                        <div class="form-text">Минимум 6 символов</div>
                    </div>
                    
                    <div class="d-grid gap-2">
                        <button type="submit" class="btn btn-primary">
                            <i class="bi bi-check-circle"></i> Сохранить изменения
                        </button>
                        <a href="/admin/users.php" class="btn btn-outline-secondary">
                            <i class="bi bi-arrow-left"></i> Назад к списку
                        </a>
                    </div>
                    
                    <div id="feedback" class="mt-3"></div>
                </form>
            </div>
        </div>
        
        <div class="col-md-6">
            <div class="glass p-4 card-ui fade-in">
                <h5 class="fw-bold">
                    <i class="bi bi-info-circle"></i> Информация о пользователе
                </h5>
                
                <div class="list-group list-group-flush">
                    <div class="list-group-item">
                        <strong>ID:</strong> <?= e($user['id']) ?>
                    </div>
                    <div class="list-group-item">
                        <strong>Зарегистрирован:</strong> <?= date('d.m.Y H:i', strtotime($user['created_at'])) ?>
                    </div>
                    <div class="list-group-item">
                        <strong>Email:</strong> <?= e($user['email']) ?>
                    </div>
                    <div class="list-group-item">
                        <strong>Имя:</strong> <?= e($user['name']) ?>
                    </div>
                    <div class="list-group-item">
                        <strong>Телефон:</strong> <?= e($user['phone'] ?? 'не указан') ?>
                    </div>
                    <div class="list-group-item">
                        <strong>Роль:</strong> 
                        <span class="badge <?= $user['role'] === 'admin' ? 'bg-danger' : 'bg-primary' ?>">
                            <?= e($user['role'] === 'admin' ? 'Администратор' : 'Пользователь') ?>
                        </span>
                    </div>
                    <div class="list-group-item">
                        <strong>Уведомления:</strong> 
                        <span class="badge <?= $user['notifications'] ? 'bg-success' : 'bg-secondary' ?>">
                            <?= $user['notifications'] ? 'Включены' : 'Выключены' ?>
                        </span>
                    </div>
                </div>
                
                <div class="alert alert-warning mt-4">
                    <i class="bi bi-exclamation-triangle"></i>
                    <strong>Внимание:</strong> Изменение данных пользователя может повлиять на его доступ к системе.
                </div>
            </div>
        </div>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const form = document.getElementById('edit-user-form');
    const feedback = document.getElementById('feedback');
    const phoneInput = document.getElementById('phone');

    if (phoneInput) {
        phoneInput.addEventListener('input', function(e) {
            let value = this.value.replace(/\D/g, '');
            
            if (value.length > 0) {
                if (value.startsWith('7')) {
                    value = '+7 ' + value.substring(1);
                } else if (value.startsWith('8')) {
                    value = '+7 ' + value.substring(1);
                } else {
                    value = '+7 ' + value;
                }
                
                value = value.substring(0, 15);
                
                let formatted = '';
                for (let i = 0; i < value.length; i++) {
                    if (i === 2 || i === 6 || i === 10 || i === 13) {
                        formatted += ' ';
                    }
                    formatted += value[i];
                }
                
                this.value = formatted.trim();
            }
        });
    }
    
    form.addEventListener('submit', function(e) {
        e.preventDefault();
        
        if (!this.checkValidity()) {
            e.stopPropagation();
            this.classList.add('was-validated');
            showFeedback('Пожалуйста, заполните все обязательные поля', 'warning');
            return;
        }
        
        const formData = new FormData(this);
        const password = formData.get('password');
        
        if (password && password.length < 6) {
            showFeedback('Пароль должен быть не менее 6 символов', 'danger');
            return;
        }
        
        showFeedback(
            '<div class="text-center">' +
            '<i class="bi bi-hourglass text-info fs-1"></i>' +
            '<p class="mt-2">Сохранение...</p>' +
            '</div>', 
            'info'
        );
        
        fetch('/ajax/admin/admin_update_user.php', {
            method: 'POST',
            body: formData
        })
        .then(res => {
            if (!res.ok) {
                throw new Error('Network response was not ok');
            }
            return res.json();
        })
        .then(data => {
            if (data.success) {
                showFeedback(
                    '<div class="text-center">' +
                    '<i class="bi bi-check-circle-fill text-success fs-1"></i>' +
                    '<h5 class="mt-2">Успешно!</h5>' +
                    '<p>Данные пользователя обновлены.</p>' +
                    '<div class="mt-3">' +
                    '<a href="/admin/users.php" class="btn btn-primary me-2">Вернуться к списку</a>' +
                    '<button class="btn btn-outline-primary" onclick="location.reload()">Остаться на странице</button>' +
                    '</div>' +
                    '</div>', 
                    'success'
                );

                const csrfField = form.querySelector('input[name="csrf"]');
                if (csrfField && data.csrf_token) {
                    csrfField.value = data.csrf_token;
                }
            } else {
                showFeedback(
                    '<div class="text-center">' +
                    '<i class="bi bi-exclamation-circle-fill text-danger fs-1"></i>' +
                    '<h5 class="mt-2">Ошибка</h5>' +
                    '<p>' + (data.error || 'Ошибка сохранения') + '</p>' +
                    '</div>', 
                    'danger'
                );
            }
        })
        .catch(error => {
            console.error('Error:', error);
            showFeedback(
                '<div class="text-center">' +
                '<i class="bi bi-exclamation-circle-fill text-danger fs-1"></i>' +
                '<h5 class="mt-2">Ошибка соединения</h5>' +
                '<p>Проверьте подключение к интернету и попробуйте снова</p>' +
                '</div>', 
                'danger'
            );
        });
    });
    
    function showFeedback(message, type = 'info') {
        feedback.innerHTML = '<div class="alert alert-' + type + ' animate__animated animate__fadeIn">' + message + '</div>';
    }

    (function() {
        'use strict';
        const forms = document.querySelectorAll('.needs-validation');
        Array.from(forms).forEach(form => {
            form.addEventListener('submit', event => {
                if (!form.checkValidity()) {
                    event.preventDefault();
                    event.stopPropagation();
                }
                form.classList.add('was-validated');
            }, false);
        });
    })();
});
</script>

<?php 
require_once __DIR__ . '/../includes/footer.php'; 
?>