<?php
// =========================
// public/register.php
// Страница регистрации пользователя
// =========================

require_once __DIR__ . '/../config/bootstrap.php';
require_once __DIR__ . '/../config/auth.php';
require_once __DIR__ . '/../config/functions.php';

// Если пользователь уже авторизован, редирект на профиль
if (is_logged_in()) {
    redirect('/profile.php');
}

$errors = [];

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Проверка CSRF
    $csrf_token = $_POST['csrf_token'] ?? '';
    if (!check_csrf($csrf_token)) {
        $errors[] = 'Ошибка безопасности. Попробуйте снова.';
    } else {
        // Получаем данные
        $name = trim($_POST['name'] ?? '');
        $email = trim($_POST['email'] ?? '');
        $password = $_POST['password'] ?? '';
        $confirm_password = $_POST['confirm_password'] ?? '';

        // Валидация
        if (empty($name) || empty($email) || empty($password) || empty($confirm_password)) {
            $errors[] = 'Заполните все поля.';
        } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            $errors[] = 'Введите корректный email.';
        } elseif ($password !== $confirm_password) {
            $errors[] = 'Пароли не совпадают.';
        } elseif (strlen($password) < 6) {
            $errors[] = 'Пароль должен быть не менее 6 символов.';
        } else {
            // Регистрация пользователя
            $res = register_user($name, $email, $password);
            if ($res['success']) {
                // Автовход после регистрации
                login_user($email, $password);
                redirect('/profile.php');
            } else {
                $errors[] = $res['error'] ?? 'Ошибка регистрации';
            }
        }
    }
}

// Подключаем header
include __DIR__ . '/../includes/header.php';
?>

<div class="container py-5">
    <div class="row justify-content-center">
        <div class="col-md-6 col-lg-5">
            <div class="card shadow-sm border-0 animate__animated animate__fadeInDown">
                <div class="card-body p-4">
                    <h3 class="card-title mb-4 text-center fw-bold">Регистрация в PhotoPrint</h3>

                    <?php
                    // Вывод ошибок
                    if ($errors) {
                        foreach ($errors as $error) {
                            echo '<div class="alert alert-danger mb-2">' . e($error) . '</div>';
                        }
                    }
                    ?>

                    <form method="POST" class="needs-validation" novalidate>
                        <!-- CSRF токен -->
                        <input type="hidden" name="csrf_token" value="<?= e($_SESSION['csrf_token'] ?? '') ?>">

                        <div class="mb-3">
                            <label for="name" class="form-label">Имя</label>
                            <input type="text" class="form-control" id="name" name="name" required 
                                   value="<?= e($_POST['name'] ?? '') ?>">
                            <div class="invalid-feedback">
                                Введите имя
                            </div>
                        </div>

                        <div class="mb-3">
                            <label for="email" class="form-label">Email</label>
                            <input type="email" class="form-control" id="email" name="email" required 
                                   value="<?= e($_POST['email'] ?? '') ?>">
                            <div class="invalid-feedback">
                                Введите корректный email
                            </div>
                        </div>

                        <div class="mb-3">
                            <label for="password" class="form-label">Пароль</label>
                            <input type="password" class="form-control" id="password" name="password" required
                                   minlength="6">
                            <div class="invalid-feedback">
                                Пароль должен быть не менее 6 символов
                            </div>
                        </div>

                        <div class="mb-3">
                            <label for="confirm_password" class="form-label">Повторите пароль</label>
                            <input type="password" class="form-control" id="confirm_password" name="confirm_password" required>
                            <div class="invalid-feedback">
                                Повторите пароль
                            </div>
                        </div>

                        <button type="submit" class="btn btn-primary w-100 mt-3">Зарегистрироваться</button>
                    </form>

                    <div class="text-center mt-3">
                        <small>Уже есть аккаунт? <a href="/login.php">Войти</a></small>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
// Валидация формы на клиенте
document.addEventListener('DOMContentLoaded', function() {
    const forms = document.querySelectorAll('.needs-validation');
    
    Array.from(forms).forEach(form => {
        form.addEventListener('submit', event => {
            if (!form.checkValidity()) {
                event.preventDefault();
                event.stopPropagation();
            }
            
            // Проверка совпадения паролей
            const password = document.getElementById('password');
            const confirmPassword = document.getElementById('confirm_password');
            
            if (password.value !== confirmPassword.value) {
                confirmPassword.setCustomValidity('Пароли не совпадают');
                event.preventDefault();
                event.stopPropagation();
            } else {
                confirmPassword.setCustomValidity('');
            }
            
            form.classList.add('was-validated');
        }, false);
    });
});
</script>

<?php
// Подключаем footer
include __DIR__ . '/../includes/footer.php';
?>