<?php
require_once '../includes/config.php';
require_once '../includes/functions.php';

if (!isAdmin()) {
    header('Location: /login.php');
    exit;
}

$title = "Панель управления";
include '../templates/admin/header.php';
include '../templates/admin/sidebar.php';

$db = db();

// Получаем статистику
$stats = [
    'users' => $db->query("SELECT COUNT(*) as count FROM users")->fetch_assoc()['count'],
    'products' => $db->query("SELECT COUNT(*) as count FROM products WHERE status = 'active'")->fetch_assoc()['count'],
    'orders_today' => $db->query("SELECT COUNT(*) as count FROM orders WHERE DATE(created_at) = CURDATE()")->fetch_assoc()['count'],
    'revenue_today' => $db->query("SELECT COALESCE(SUM(total_amount), 0) as amount FROM orders WHERE DATE(created_at) = CURDATE() AND payment_status = 'paid'")->fetch_assoc()['amount'],
    'pending_orders' => $db->query("SELECT COUNT(*) as count FROM orders WHERE payment_status = 'pending'")->fetch_assoc()['count'],
    'moderation_queue' => $db->query("SELECT COUNT(*) as count FROM products WHERE requires_moderation = 1")->fetch_assoc()['count']
];

// Последние заказы
$recentOrders = $db->query("
    SELECT o.*, u.username, u.email 
    FROM orders o 
    LEFT JOIN users u ON o.user_id = u.id 
    ORDER BY o.created_at DESC 
    LIMIT 10
");

// График продаж за последние 7 дней
$salesChart = [];
$result = $db->query("
    SELECT 
        DATE(created_at) as date,
        COUNT(*) as orders,
        SUM(total_amount) as revenue
    FROM orders 
    WHERE payment_status = 'paid' 
    AND created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY)
    GROUP BY DATE(created_at)
    ORDER BY date
");

while ($row = $result->fetch_assoc()) {
    $salesChart[] = $row;
}
?>

<div class="container-fluid">
    <h1 class="h3 mb-4">Панель управления</h1>
    
    <!-- Статистика -->
    <div class="row mb-4">
        <div class="col-xl-3 col-md-6 mb-4">
            <div class="card border-left-primary shadow h-100 py-2">
                <div class="card-body">
                    <div class="row no-gutters align-items-center">
                        <div class="col mr-2">
                            <div class="text-xs font-weight-bold text-primary text-uppercase mb-1">
                                Пользователи</div>
                            <div class="h5 mb-0 font-weight-bold text-gray-800">
                                <?php echo number_format($stats['users']); ?>
                            </div>
                        </div>
                        <div class="col-auto">
                            <i class="fas fa-users fa-2x text-gray-300"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        
        <div class="col-xl-3 col-md-6 mb-4">
            <div class="card border-left-success shadow h-100 py-2">
                <div class="card-body">
                    <div class="row no-gutters align-items-center">
                        <div class="col mr-2">
                            <div class="text-xs font-weight-bold text-success text-uppercase mb-1">
                                Товары</div>
                            <div class="h5 mb-0 font-weight-bold text-gray-800">
                                <?php echo number_format($stats['products']); ?>
                            </div>
                        </div>
                        <div class="col-auto">
                            <i class="fas fa-box fa-2x text-gray-300"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        
        <div class="col-xl-3 col-md-6 mb-4">
            <div class="card border-left-info shadow h-100 py-2">
                <div class="card-body">
                    <div class="row no-gutters align-items-center">
                        <div class="col mr-2">
                            <div class="text-xs font-weight-bold text-info text-uppercase mb-1">
                                Заказов сегодня</div>
                            <div class="h5 mb-0 font-weight-bold text-gray-800">
                                <?php echo number_format($stats['orders_today']); ?>
                            </div>
                        </div>
                        <div class="col-auto">
                            <i class="fas fa-shopping-cart fa-2x text-gray-300"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        
        <div class="col-xl-3 col-md-6 mb-4">
            <div class="card border-left-warning shadow h-100 py-2">
                <div class="card-body">
                    <div class="row no-gutters align-items-center">
                        <div class="col mr-2">
                            <div class="text-xs font-weight-bold text-warning text-uppercase mb-1">
                                Выручка сегодня</div>
                            <div class="h5 mb-0 font-weight-bold text-gray-800">
                                <?php echo number_format($stats['revenue_today'], 2); ?> ₽
                            </div>
                        </div>
                        <div class="col-auto">
                            <i class="fas fa-ruble-sign fa-2x text-gray-300"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <div class="row">
        <!-- График продаж -->
        <div class="col-xl-8 col-lg-7">
            <div class="card shadow mb-4">
                <div class="card-header py-3">
                    <h6 class="m-0 font-weight-bold text-primary">Продажи за 7 дней</h6>
                </div>
                <div class="card-body">
                    <div class="chart-area">
                        <canvas id="salesChart"></canvas>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Последние заказы -->
        <div class="col-xl-4 col-lg-5">
            <div class="card shadow mb-4">
                <div class="card-header py-3">
                    <h6 class="m-0 font-weight-bold text-primary">Последние заказы</h6>
                </div>
                <div class="card-body">
                    <div class="table-responsive">
                        <table class="table table-sm">
                            <thead>
                                <tr>
                                    <th>№</th>
                                    <th>Сумма</th>
                                    <th>Статус</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php while ($order = $recentOrders->fetch_assoc()): ?>
                                <tr>
                                    <td>
                                        <a href="orders.php?view=<?php echo $order['id']; ?>">
                                            <?php echo $order['order_number']; ?>
                                        </a>
                                        <br>
                                        <small><?php echo date('d.m H:i', strtotime($order['created_at'])); ?></small>
                                    </td>
                                    <td><?php echo number_format($order['total_amount'], 2); ?> ₽</td>
                                    <td>
                                        <span class="badge badge-<?php 
                                            echo $order['payment_status'] === 'paid' ? 'success' : 
                                                 ($order['payment_status'] === 'pending' ? 'warning' : 'danger');
                                        ?>">
                                            <?php echo $order['payment_status']; ?>
                                        </span>
                                    </td>
                                </tr>
                                <?php endwhile; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
            
            <!-- Быстрые действия -->
            <div class="card shadow">
                <div class="card-header py-3">
                    <h6 class="m-0 font-weight-bold text-primary">Быстрые действия</h6>
                </div>
                <div class="card-body">
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <a href="products.php" class="btn btn-primary btn-block">
                                <i class="fas fa-box mr-2"></i>Товары
                            </a>
                        </div>
                        <div class="col-md-6 mb-3">
                            <a href="orders.php" class="btn btn-success btn-block">
                                <i class="fas fa-shopping-cart mr-2"></i>Заказы
                            </a>
                        </div>
                        <div class="col-md-6 mb-3">
                            <a href="moderation.php" class="btn btn-warning btn-block">
                                <i class="fas fa-check-circle mr-2"></i>Модерация
                                <?php if ($stats['moderation_queue'] > 0): ?>
                                <span class="badge badge-light"><?php echo $stats['moderation_queue']; ?></span>
                                <?php endif; ?>
                            </a>
                        </div>
                        <div class="col-md-6 mb-3">
                            <a href="users.php" class="btn btn-info btn-block">
                                <i class="fas fa-users mr-2"></i>Пользователи
                            </a>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Chart.js для графиков -->
<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<script>
// График продаж
const salesCtx = document.getElementById('salesChart').getContext('2d');
const salesChart = new Chart(salesCtx, {
    type: 'line',
    data: {
        labels: [
            <?php 
            $dates = array_column($salesChart, 'date');
            foreach ($dates as $date) {
                echo "'" . date('d.m', strtotime($date)) . "', ";
            }
            ?>
        ],
        datasets: [{
            label: 'Выручка (₽)',
            data: [
                <?php 
                $revenues = array_column($salesChart, 'revenue');
                foreach ($revenues as $revenue) {
                    echo $revenue ?: 0 . ', ';
                }
                ?>
            ],
            borderColor: 'rgb(54, 162, 235)',
            backgroundColor: 'rgba(54, 162, 235, 0.1)',
            tension: 0.3
        }, {
            label: 'Заказы',
            data: [
                <?php 
                $orders = array_column($salesChart, 'orders');
                foreach ($orders as $order) {
                    echo $order ?: 0 . ', ';
                }
                ?>
            ],
            borderColor: 'rgb(75, 192, 192)',
            backgroundColor: 'rgba(75, 192, 192, 0.1)',
            tension: 0.3
        }]
    },
    options: {
        responsive: true,
        plugins: {
            legend: {
                position: 'top',
            }
        },
        scales: {
            y: {
                beginAtZero: true
            }
        }
    }
});
</script>

<?php include '../templates/admin/footer.php'; ?>