<?php
require_once '../includes/config.php';
require_once '../includes/functions.php';
require_once '../includes/payment/yoomoney.php';
require_once '../includes/payment/azvox.php';

header('Content-Type: application/json');

$input = file_get_contents('php://input');
$data = json_decode($input, true);

if (!$data) {
    $data = $_POST;
}

$log = [
    'timestamp' => date('Y-m-d H:i:s'),
    'ip' => $_SERVER['REMOTE_ADDR'],
    'data' => $data
];

file_put_contents('../logs/payments.log', json_encode($log) . PHP_EOL, FILE_APPEND);

// Определяем платежную систему
if (isset($data['notification_type']) && $data['notification_type'] === 'card-incoming') {
    // ЮMoney Webhook
    handleYooMoneyCallback($data);
} elseif (isset($data['azvox_signature'])) {
    // Azvox Webhook
    handleAzvoxCallback($data);
} else {
    // Локальная обработка (для тестов)
    handleTestCallback($data);
}

function handleYooMoneyCallback($data) {
    $db = db();
    
    // Проверяем подпись
    if (!YooMoneyPayment::verifyNotification($data)) {
        http_response_code(400);
        echo json_encode(['error' => 'Invalid signature']);
        return;
    }
    
    $orderId = $data['label'];
    $amount = $data['amount'];
    $paymentId = $data['operation_id'];
    
    // Находим заказ
    $stmt = $db->prepare("
        SELECT o.*, u.email, u.username 
        FROM orders o 
        LEFT JOIN users u ON o.user_id = u.id 
        WHERE o.order_number = ? AND o.payment_status = 'pending'
    ");
    $stmt->bind_param("s", $orderId);
    $stmt->execute();
    $order = $stmt->get_result()->fetch_assoc();
    
    if (!$order) {
        http_response_code(404);
        echo json_encode(['error' => 'Order not found']);
        return;
    }
    
    // Проверяем сумму
    if ($amount != $order['total_amount']) {
        http_response_code(400);
        echo json_encode(['error' => 'Amount mismatch']);
        return;
    }
    
    // Обновляем статус заказа
    $db->begin_transaction();
    
    try {
        $stmt = $db->prepare("
            UPDATE orders 
            SET payment_status = 'paid', 
                status = 'processing',
                payment_id = ?,
                updated_at = NOW() 
            WHERE order_number = ?
        ");
        $stmt->bind_param("ss", $paymentId, $orderId);
        $stmt->execute();
        
        // Записываем платеж
        $stmt = $db->prepare("
            INSERT INTO payments (order_id, amount, currency, method, transaction_id, status, created_at)
            VALUES (?, ?, 'RUB', 'yoomoney', ?, 'success', NOW())
        ");
        $stmt->bind_param("ids", $order['id'], $amount, $paymentId);
        $stmt->execute();
        
        // Генерируем токен для скачивания
        $downloadToken = bin2hex(random_bytes(32));
        $tokenExpires = date('Y-m-d H:i:s', strtotime('+7 days'));
        
        $stmt = $db->prepare("
            UPDATE orders 
            SET download_token = ?, token_expires = ? 
            WHERE id = ?
        ");
        $stmt->bind_param("ssi", $downloadToken, $tokenExpires, $order['id']);
        $stmt->execute();
        
        // Отправляем email пользователю
        $downloadLink = SITE_URL . "/download.php?token=" . $downloadToken;
        $subject = "Ваш заказ #" . $orderId . " оплачен";
        $message = "
            <h2>Спасибо за покупку!</h2>
            <p>Ваш заказ #{$orderId} успешно оплачен.</p>
            <p>Ссылка для скачивания: <a href='{$downloadLink}'>{$downloadLink}</a></p>
            <p>Ссылка действительна до: {$tokenExpires}</p>
        ";
        
        sendEmail($order['email'], $subject, $message);
        
        $db->commit();
        
        // Ответ для ЮMoney
        echo json_encode(['success' => true]);
        
    } catch (Exception $e) {
        $db->rollback();
        http_response_code(500);
        echo json_encode(['error' => 'Payment processing failed']);
    }
}

function handleAzvoxCallback($data) {
    // Аналогичная обработка для Azvox
    // Проверяем подпись, обновляем заказ и т.д.
}

function handleTestCallback($data) {
    // Для тестирования без реальных платежных систем
    if (isset($data['order_id']) && isset($data['success']) && $data['success'] === true) {
        $db = db();
        
        $stmt = $db->prepare("
            UPDATE orders 
            SET payment_status = 'paid', 
                status = 'completed',
                updated_at = NOW() 
            WHERE order_number = ?
        ");
        $stmt->bind_param("s", $data['order_id']);
        $stmt->execute();
        
        echo json_encode(['success' => true]);
    } else {
        echo json_encode(['error' => 'Invalid test callback']);
    }
}
?>