// Админ-скрипты
document.addEventListener('DOMContentLoaded', function() {
    // Инициализация всех административных функций
    
    // 1. Инициализация таблиц с сортировкой
    initSortableTables();
    
    // 2. Инициализация фильтров
    initAdminFilters();
    
    // 3. Инициализация подтверждений
    initConfirmations();
    
    // 4. Инициализация массовых действий
    initBulkActions();
    
    // 5. Инициализация инлайн-редактирования
    initInlineEditing();
    
    // 6. Инициализация загрузки файлов
    initFileUploads();
    
    // 7. Инициализация live search
    initLiveSearch();
    
    // 8. Инициализация уведомлений
    initAdminNotifications();
});

// Сортировка таблиц
function initSortableTables() {
    document.querySelectorAll('table.sortable th[data-sort]').forEach(header => {
        header.style.cursor = 'pointer';
        header.addEventListener('click', function() {
            const table = this.closest('table');
            const column = this.getAttribute('data-sort');
            const direction = this.getAttribute('data-direction') === 'asc' ? 'desc' : 'asc';
            
            // Обновляем направление
            this.setAttribute('data-direction', direction);
            
            // Удаляем сортировку с других заголовков
            table.querySelectorAll('th[data-sort]').forEach(th => {
                if (th !== this) {
                    th.removeAttribute('data-direction');
                    th.classList.remove('sorting-asc', 'sorting-desc');
                }
            });
            
            // Добавляем классы
            this.classList.remove('sorting-asc', 'sorting-desc');
            this.classList.add(`sorting-${direction}`);
            
            // Сортируем таблицу
            sortTable(table, column, direction);
        });
    });
}

function sortTable(table, column, direction) {
    const tbody = table.querySelector('tbody');
    const rows = Array.from(tbody.querySelectorAll('tr'));
    
    rows.sort((a, b) => {
        const aValue = a.querySelector(`td:nth-child(${getColumnIndex(table, column)})`)?.textContent || '';
        const bValue = b.querySelector(`td:nth-child(${getColumnIndex(table, column)})`)?.textContent || '';
        
        // Пытаемся определить тип данных
        const aNum = parseFloat(aValue.replace(/[^\d.-]/g, ''));
        const bNum = parseFloat(bValue.replace(/[^\d.-]/g, ''));
        
        if (!isNaN(aNum) && !isNaN(bNum)) {
            // Числовое сравнение
            return direction === 'asc' ? aNum - bNum : bNum - aNum;
        } else {
            // Строковое сравнение
            return direction === 'asc' 
                ? aValue.localeCompare(bValue)
                : bValue.localeCompare(aValue);
        }
    });
    
    // Удаляем старые строки
    while (tbody.firstChild) {
        tbody.removeChild(tbody.firstChild);
    }
    
    // Добавляем отсортированные строки
    rows.forEach(row => tbody.appendChild(row));
}

function getColumnIndex(table, columnName) {
    const headers = table.querySelectorAll('th[data-sort]');
    for (let i = 0; i < headers.length; i++) {
        if (headers[i].getAttribute('data-sort') === columnName) {
            return i + 1;
        }
    }
    return 1;
}

// Фильтры
function initAdminFilters() {
    document.querySelectorAll('.admin-filter').forEach(filter => {
        const input = filter.querySelector('input, select');
        const table = filter.closest('.table-responsive')?.querySelector('table') || 
                     document.querySelector('table');
        
        if (input && table) {
            input.addEventListener('input', debounce(function() {
                filterTable(table, this.value.toLowerCase());
            }, 300));
        }
    });
}

function filterTable(table, searchText) {
    const rows = table.querySelectorAll('tbody tr');
    let visibleCount = 0;
    
    rows.forEach(row => {
        const text = row.textContent.toLowerCase();
        const isVisible = text.includes(searchText);
        
        row.style.display = isVisible ? '' : 'none';
        if (isVisible) visibleCount++;
    });
    
    // Показываем/скрываем сообщение "нет результатов"
    const noResults = table.parentNode.querySelector('.no-results');
    if (noResults) {
        noResults.style.display = visibleCount === 0 ? '' : 'none';
    }
}

// Подтверждения действий
function initConfirmations() {
    document.addEventListener('click', function(e) {
        const confirmBtn = e.target.closest('[data-confirm]');
        if (confirmBtn) {
            e.preventDefault();
            
            const message = confirmBtn.getAttribute('data-confirm') || 'Вы уверены?';
            const form = confirmBtn.closest('form');
            const href = confirmBtn.getAttribute('href');
            
            if (confirm(message)) {
                if (form) {
                    form.submit();
                } else if (href) {
                    window.location.href = href;
                }
            }
        }
    });
}

// Массовые действия
function initBulkActions() {
    const selectAll = document.getElementById('selectAll');
    if (selectAll) {
        selectAll.addEventListener('change', function() {
            const checkboxes = document.querySelectorAll('input[name="selected[]"]');
            checkboxes.forEach(cb => cb.checked = this.checked);
            updateBulkActions();
        });
    }
    
    document.querySelectorAll('input[name="selected[]"]').forEach(checkbox => {
        checkbox.addEventListener('change', updateBulkActions);
    });
    
    document.querySelectorAll('.bulk-action-form').forEach(form => {
        const select = form.querySelector('select[name="action"]');
        const button = form.querySelector('button[type="submit"]');
        
        if (select && button) {
            select.addEventListener('change', function() {
                button.textContent = this.options[this.selectedIndex].text;
            });
        }
    });
}

function updateBulkActions() {
    const selected = document.querySelectorAll('input[name="selected[]"]:checked').length;
    const bulkActions = document.querySelector('.bulk-actions');
    
    if (bulkActions) {
        bulkActions.style.display = selected > 0 ? 'block' : 'none';
        bulkActions.querySelector('.selected-count').textContent = selected;
    }
}

// Инлайн-редактирование
function initInlineEditing() {
    document.querySelectorAll('.editable').forEach(element => {
        element.addEventListener('dblclick', function() {
            if (this.hasAttribute('data-editing')) return;
            
            const originalValue = this.textContent.trim();
            const fieldName = this.getAttribute('data-field');
            const id = this.getAttribute('data-id');
            
            // Создаем input
            const input = document.createElement('input');
            input.type = 'text';
            input.className = 'form-control form-control-sm';
            input.value = originalValue;
            
            // Заменяем содержимое
            this.setAttribute('data-editing', 'true');
            this.innerHTML = '';
            this.appendChild(input);
            input.focus();
            input.select();
            
            // Обработка завершения редактирования
            function finishEdit() {
                const newValue = input.value.trim();
                
                if (newValue !== originalValue && newValue !== '') {
                    // Отправка на сервер
                    fetch('/api/admin/update.php', {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json',
                            'X-CSRF-Token': window.App.csrfToken
                        },
                        body: JSON.stringify({
                            id: id,
                            field: fieldName,
                            value: newValue,
                            table: element.closest('table').getAttribute('data-table')
                        })
                    })
                    .then(response => response.json())
                    .then(data => {
                        if (data.success) {
                            element.textContent = newValue;
                            showAdminNotification('Изменения сохранены', 'success');
                        } else {
                            element.textContent = originalValue;
                            showAdminNotification('Ошибка сохранения: ' + data.error, 'error');
                        }
                    })
                    .catch(error => {
                        element.textContent = originalValue;
                        showAdminNotification('Ошибка сети', 'error');
                    });
                } else {
                    element.textContent = originalValue;
                }
                
                element.removeAttribute('data-editing');
            }
            
            input.addEventListener('blur', finishEdit);
            input.addEventListener('keypress', function(e) {
                if (e.key === 'Enter') {
                    finishEdit();
                } else if (e.key === 'Escape') {
                    element.textContent = originalValue;
                    element.removeAttribute('data-editing');
                }
            });
        });
    });
}

// Загрузка файлов
function initFileUploads() {
    document.querySelectorAll('.file-upload').forEach(upload => {
        const input = upload.querySelector('input[type="file"]');
        const preview = upload.querySelector('.file-preview');
        const removeBtn = upload.querySelector('.file-remove');
        
        if (input) {
            input.addEventListener('change', function() {
                if (this.files && this.files[0]) {
                    const file = this.files[0];
                    
                    // Показываем превью для изображений
                    if (file.type.startsWith('image/') && preview) {
                        const reader = new FileReader();
                        reader.onload = function(e) {
                            preview.innerHTML = `<img src="${e.target.result}" alt="Preview" style="max-width: 200px;">`;
                        };
                        reader.readAsDataURL(file);
                    }
                    
                    // Показываем информацию о файле
                    const info = upload.querySelector('.file-info');
                    if (info) {
                        info.textContent = `${file.name} (${(file.size / 1024).toFixed(1)} KB)`;
                    }
                }
            });
        }
        
        if (removeBtn) {
            removeBtn.addEventListener('click', function() {
                if (input) input.value = '';
                if (preview) preview.innerHTML = '';
                const info = upload.querySelector('.file-info');
                if (info) info.textContent = 'Файл не выбран';
            });
        }
    });
}

// Live search
function initLiveSearch() {
    document.querySelectorAll('.live-search').forEach(search => {
        const input = search.querySelector('input');
        const results = search.querySelector('.search-results');
        
        if (input && results) {
            input.addEventListener('input', debounce(function() {
                const query = this.value.trim();
                
                if (query.length < 2) {
                    results.style.display = 'none';
                    return;
                }
                
                fetch(`/api/admin/search.php?q=${encodeURIComponent(query)}`, {
                    headers: {
                        'X-CSRF-Token': window.App.csrfToken
                    }
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success && data.results.length > 0) {
                        results.innerHTML = data.results.map(item => `
                            <a href="${item.url}" class="search-result-item">
                                <div class="search-result-title">${item.title}</div>
                                <div class="search-result-description">${item.description || ''}</div>
                            </a>
                        `).join('');
                        results.style.display = 'block';
                    } else {
                        results.innerHTML = '<div class="search-result-item text-muted">Ничего не найдено</div>';
                        results.style.display = 'block';
                    }
                })
                .catch(error => {
                    results.style.display = 'none';
                });
            }, 500));
            
            // Скрываем результаты при клике вне
            document.addEventListener('click', function(e) {
                if (!search.contains(e.target)) {
                    results.style.display = 'none';
                }
            });
        }
    });
}

// Уведомления в админке
function initAdminNotifications() {
    // Периодическая проверка новых уведомлений
    if (window.App?.userRole === 'admin' || window.App?.userRole === 'moderator') {
        setInterval(checkAdminNotifications, 60000); // Каждую минуту
    }
}

function checkAdminNotifications() {
    fetch('/api/admin/notifications.php', {
        headers: {
            'X-CSRF-Token': window.App.csrfToken
        }
    })
    .then(response => response.json())
    .then(data => {
        if (data.success && data.count > 0) {
            updateNotificationBadge(data.count);
            
            // Показываем тост если есть новые
            if (data.new > 0) {
                showAdminNotification(`У вас ${data.new} новых уведомлений`, 'info');
            }
        }
    })
    .catch(() => {});
}

function updateNotificationBadge(count) {
    const badge = document.querySelector('#adminNotificationsBadge');
    if (badge) {
        badge.textContent = count;
        badge.style.display = count > 0 ? '' : 'none';
    }
}

function showAdminNotification(message, type = 'info') {
    const notification = document.createElement('div');
    notification.className = `admin-notification alert alert-${type} alert-dismissible fade show`;
    notification.innerHTML = `
        ${message}
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    `;
    
    const container = document.getElementById('adminNotifications') || 
                     document.querySelector('.admin-notifications') ||
                     document.body;
    
    // Удаляем старые уведомления
    const oldNotifications = container.querySelectorAll('.admin-notification');
    if (oldNotifications.length > 3) {
        oldNotifications[0].remove();
    }
    
    container.appendChild(notification);
    
    // Автоудаление через 5 секунд
    setTimeout(() => {
        if (notification.parentNode) {
            notification.remove();
        }
    }, 5000);
}

// Вспомогательные функции
function debounce(func, wait) {
    let timeout;
    return function executedFunction(...args) {
        const later = () => {
            clearTimeout(timeout);
            func(...args);
        };
        clearTimeout(timeout);
        timeout = setTimeout(later, wait);
    };
}

// Экспорт функций
if (typeof window !== 'undefined') {
    window.Admin = {
        showNotification: showAdminNotification,
        sortTable: sortTable,
        filterTable: filterTable,
        updateBulkActions: updateBulkActions,
        checkNotifications: checkAdminNotifications
    };
}