<?php
require_once 'includes/config.php';
require_once 'includes/functions.php';

$title = "Каталог товаров";
$theme = isset($_COOKIE['theme']) ? $_COOKIE['theme'] : 'light';

// Параметры фильтрации
$category_id = isset($_GET['category']) ? (int)$_GET['category'] : 0;
$sort = isset($_GET['sort']) ? $_GET['sort'] : 'newest';
$min_price = isset($_GET['min_price']) ? (float)$_GET['min_price'] : 0;
$max_price = isset($_GET['max_price']) ? (float)$_GET['max_price'] : 100000;
$search = isset($_GET['search']) ? sanitize($_GET['search']) : '';
$page = isset($_GET['page']) ? max(1, (int)$_GET['page']) : 1;
$limit = 12;
$offset = ($page - 1) * $limit;

// Получаем товары с фильтрами
$db = db();
$where = [];
$params = [];
$types = '';

// Фильтр по категории
if ($category_id > 0) {
    $where[] = "p.category_id = ?";
    $params[] = $category_id;
    $types .= 'i';
}

// Фильтр по цене
if ($min_price > 0) {
    $where[] = "COALESCE(p.discount_price, p.price) >= ?";
    $params[] = $min_price;
    $types .= 'd';
}

if ($max_price > 0 && $max_price < 100000) {
    $where[] = "COALESCE(p.discount_price, p.price) <= ?";
    $params[] = $max_price;
    $types .= 'd';
}

// Поиск
if (!empty($search)) {
    $where[] = "(p.name LIKE ? OR p.description LIKE ? OR p.short_description LIKE ?)";
    $params[] = "%$search%";
    $params[] = "%$search%";
    $params[] = "%$search%";
    $types .= 'sss';
}

// Статус товара
$where[] = "p.status = 'active'";

// Формируем запрос
$where_sql = !empty($where) ? "WHERE " . implode(" AND ", $where) : "";

// Сортировка
$order_by = "ORDER BY ";
switch ($sort) {
    case 'price_asc':
        $order_by .= "COALESCE(p.discount_price, p.price) ASC";
        break;
    case 'price_desc':
        $order_by .= "COALESCE(p.discount_price, p.price) DESC";
        break;
    case 'popular':
        $order_by .= "p.downloads_count DESC";
        break;
    case 'name':
        $order_by .= "p.name ASC";
        break;
    default: // newest
        $order_by .= "p.created_at DESC";
        break;
}

// Получаем товары
$sql = "
    SELECT SQL_CALC_FOUND_ROWS 
        p.*, 
        c.name as category_name,
        c.slug as category_slug,
        COALESCE(p.discount_price, p.price) as final_price
    FROM products p
    LEFT JOIN categories c ON p.category_id = c.id
    $where_sql
    $order_by
    LIMIT ? OFFSET ?
";

$params[] = $limit;
$params[] = $offset;
$types .= 'ii';

$stmt = $db->prepare($sql);

if (!empty($params)) {
    $stmt->bind_param($types, ...$params);
}

$stmt->execute();
$products = $stmt->get_result();

// Общее количество товаров
$total_result = $db->query("SELECT FOUND_ROWS() as total");
$total_row = $total_result->fetch_assoc();
$total_products = $total_row['total'];
$total_pages = ceil($total_products / $limit);

// Получаем категории для фильтра
$categories = $db->query("
    SELECT c.*, COUNT(p.id) as product_count
    FROM categories c
    LEFT JOIN products p ON c.id = p.category_id AND p.status = 'active'
    WHERE c.is_active = 1
    GROUP BY c.id
    ORDER BY c.sort_order, c.name
");

// Получаем диапазон цен
$price_range = $db->query("
    SELECT 
        MIN(COALESCE(discount_price, price)) as min_price,
        MAX(COALESCE(discount_price, price)) as max_price
    FROM products 
    WHERE status = 'active'
")->fetch_assoc();

include 'templates/header.php';
?>

<div class="container py-5">
    <div class="row">
        <!-- Фильтры -->
        <div class="col-lg-3 mb-4">
            <div class="card sticky-top" style="top: 20px;">
                <div class="card-header">
                    <h5 class="mb-0"><i class="fas fa-filter me-2"></i>Фильтры</h5>
                </div>
                
                <div class="card-body">
                    <!-- Категории -->
                    <div class="mb-4">
                        <h6 class="mb-3"><i class="fas fa-tags me-2"></i>Категории</h6>
                        <div class="list-group list-group-flush">
                            <a href="/catalog.php" 
                               class="list-group-item list-group-item-action d-flex justify-content-between align-items-center <?php echo $category_id == 0 ? 'active' : ''; ?>">
                                Все категории
                                <span class="badge bg-primary rounded-pill">
                                    <?php echo $total_products; ?>
                                </span>
                            </a>
                            
                            <?php while ($cat = $categories->fetch_assoc()): ?>
                            <a href="/catalog.php?category=<?php echo $cat['id']; ?>" 
                               class="list-group-item list-group-item-action d-flex justify-content-between align-items-center <?php echo $category_id == $cat['id'] ? 'active' : ''; ?>">
                                <?php echo htmlspecialchars($cat['name']); ?>
                                <span class="badge bg-secondary rounded-pill">
                                    <?php echo $cat['product_count']; ?>
                                </span>
                            </a>
                            <?php endwhile; ?>
                        </div>
                    </div>
                    
                    <!-- Цена -->
                    <div class="mb-4">
                        <h6 class="mb-3"><i class="fas fa-ruble-sign me-2"></i>Цена, ₽</h6>
                        <div class="row g-2 mb-3">
                            <div class="col">
                                <input type="number" 
                                       class="form-control form-control-sm" 
                                       id="minPrice" 
                                       placeholder="От" 
                                       value="<?php echo $min_price; ?>"
                                       min="0">
                            </div>
                            <div class="col">
                                <input type="number" 
                                       class="form-control form-control-sm" 
                                       id="maxPrice" 
                                       placeholder="До" 
                                       value="<?php echo $max_price && $max_price < 100000 ? $max_price : ''; ?>"
                                       min="0">
                            </div>
                        </div>
                        <div class="range-slider">
                            <input type="range" 
                                   class="form-range" 
                                   min="0" 
                                   max="<?php echo ceil($price_range['max_price'] / 1000) * 1000; ?>" 
                                   step="100" 
                                   value="<?php echo $min_price; ?>"
                                   id="priceRangeMin">
                            <input type="range" 
                                   class="form-range" 
                                   min="0" 
                                   max="<?php echo ceil($price_range['max_price'] / 1000) * 1000; ?>" 
                                   step="100" 
                                   value="<?php echo $max_price && $max_price < 100000 ? $max_price : $price_range['max_price']; ?>"
                                   id="priceRangeMax">
                        </div>
                        <div class="d-flex justify-content-between small text-muted">
                            <span id="priceMinLabel">0 ₽</span>
                            <span id="priceMaxLabel"><?php echo number_format($price_range['max_price'], 0); ?> ₽</span>
                        </div>
                    </div>
                    
                    <!-- Сортировка -->
                    <div class="mb-4">
                        <h6 class="mb-3"><i class="fas fa-sort me-2"></i>Сортировка</h6>
                        <select class="form-select form-select-sm" id="sortSelect">
                            <option value="newest" <?php echo $sort == 'newest' ? 'selected' : ''; ?>>Сначала новые</option>
                            <option value="price_asc" <?php echo $sort == 'price_asc' ? 'selected' : ''; ?>>Цена по возрастанию</option>
                            <option value="price_desc" <?php echo $sort == 'price_desc' ? 'selected' : ''; ?>>Цена по убыванию</option>
                            <option value="popular" <?php echo $sort == 'popular' ? 'selected' : ''; ?>>По популярности</option>
                            <option value="name" <?php echo $sort == 'name' ? 'selected' : ''; ?>>По названию (А-Я)</option>
                        </select>
                    </div>
                    
                    <!-- Кнопки фильтрации -->
                    <div class="d-grid gap-2">
                        <button class="btn btn-primary" id="applyFilters">
                            <i class="fas fa-check me-2"></i>Применить фильтры
                        </button>
                        <a href="/catalog.php" class="btn btn-outline-secondary">
                            <i class="fas fa-times me-2"></i>Сбросить фильтры
                        </a>
                    </div>
                </div>
            </div>
            
            <!-- Баннеры или реклама -->
            <div class="card mt-3">
                <div class="card-body text-center">
                    <h6 class="card-title">Скидка 20% новым покупателям</h6>
                    <p class="small text-muted">Используйте промокод: <strong>NEW20</strong></p>
                    <a href="/register.php" class="btn btn-sm btn-success">Получить скидку</a>
                </div>
            </div>
        </div>
        
        <!-- Товары -->
        <div class="col-lg-9">
            <!-- Заголовок и управление -->
            <div class="d-flex justify-content-between align-items-center mb-4">
                <div>
                    <h1 class="h3 mb-0">Каталог товаров</h1>
                    <p class="text-muted mb-0">
                        Найдено товаров: <strong><?php echo $total_products; ?></strong>
                        <?php if ($category_id > 0): ?>
                            в категории "<?php echo htmlspecialchars($categories->fetch_assoc()['name'] ?? ''); ?>"
                        <?php endif; ?>
                    </p>
                </div>
                
                <div class="d-flex align-items-center">
                    <!-- Поиск -->
                    <form class="d-flex me-3" action="/catalog.php" method="GET">
                        <div class="input-group input-group-sm">
                            <input type="text" 
                                   class="form-control" 
                                   name="search" 
                                   placeholder="Поиск товаров..." 
                                   value="<?php echo htmlspecialchars($search); ?>"
                                   style="width: 200px;">
                            <button class="btn btn-outline-primary" type="submit">
                                <i class="fas fa-search"></i>
                            </button>
                        </div>
                    </form>
                    
                    <!-- Вид отображения -->
                    <div class="btn-group btn-group-sm" role="group">
                        <input type="radio" class="btn-check" name="viewMode" id="viewGrid" autocomplete="off" checked>
                        <label class="btn btn-outline-primary" for="viewGrid" title="Плитка">
                            <i class="fas fa-th"></i>
                        </label>
                        
                        <input type="radio" class="btn-check" name="viewMode" id="viewList" autocomplete="off">
                        <label class="btn btn-outline-primary" for="viewList" title="Список">
                            <i class="fas fa-list"></i>
                        </label>
                    </div>
                </div>
            </div>
            
            <?php if ($products->num_rows > 0): ?>
                <!-- Товары в виде плитки -->
                <div class="row" id="productsGrid">
                    <?php while ($product = $products->fetch_assoc()): ?>
                    <div class="col-xl-4 col-md-6 mb-4">
                        <?php 
                        // Включаем шаблон карточки товара
                        include 'templates/product-card.php';
                        ?>
                    </div>
                    <?php endwhile; ?>
                </div>
                
                <!-- Товары в виде списка (скрыто по умолчанию) -->
                <div class="d-none" id="productsList">
                    <?php 
                    // Сброс указателя результата
                    $products->data_seek(0);
                    while ($product = $products->fetch_assoc()): 
                    ?>
                    <div class="card mb-3">
                        <div class="row g-0">
                            <div class="col-md-4">
                                <div class="card-img-container" style="height: 200px;">
                                    <img src="<?php echo $product['image'] ?: '/assets/images/default-product.jpg'; ?>" 
                                         class="img-fluid rounded-start h-100 w-100" 
                                         alt="<?php echo htmlspecialchars($product['name']); ?>"
                                         style="object-fit: cover;">
                                </div>
                            </div>
                            <div class="col-md-8">
                                <div class="card-body">
                                    <h5 class="card-title">
                                        <a href="/product.php?id=<?php echo $product['id']; ?>" class="text-decoration-none">
                                            <?php echo htmlspecialchars($product['name']); ?>
                                        </a>
                                    </h5>
                                    <p class="card-text text-muted small">
                                        <i class="fas fa-tag me-1"></i>
                                        <a href="/catalog.php?category=<?php echo $product['category_id']; ?>" class="text-decoration-none">
                                            <?php echo htmlspecialchars($product['category_name']); ?>
                                        </a>
                                    </p>
                                    <p class="card-text"><?php echo htmlspecialchars($product['short_description'] ?? substr($product['description'], 0, 200)); ?></p>
                                    
                                    <div class="d-flex justify-content-between align-items-center mt-3">
                                        <div class="price">
                                            <?php if ($product['discount_price']): ?>
                                                <div class="text-muted small text-decoration-line-through">
                                                    <?php echo number_format($product['price'], 2); ?> ₽
                                                </div>
                                                <div class="h4 text-success mb-0">
                                                    <?php echo number_format($product['discount_price'], 2); ?> ₽
                                                </div>
                                            <?php else: ?>
                                                <div class="h4 text-primary mb-0">
                                                    <?php echo number_format($product['price'], 2); ?> ₽
                                                </div>
                                            <?php endif; ?>
                                        </div>
                                        
                                        <div class="btn-group">
                                            <a href="/product.php?id=<?php echo $product['id']; ?>" class="btn btn-outline-primary">
                                                <i class="fas fa-eye me-1"></i>Подробнее
                                            </a>
                                            <button class="btn btn-primary add-to-cart" data-product-id="<?php echo $product['id']; ?>">
                                                <i class="fas fa-cart-plus me-1"></i>В корзину
                                            </button>
                                        </div>
                                    </div>
                                    
                                    <div class="mt-3 small text-muted">
                                        <i class="fas fa-download me-1"></i>
                                        <?php echo number_format($product['downloads_count']); ?> скачиваний
                                        <span class="mx-2">•</span>
                                        <i class="fas fa-calendar me-1"></i>
                                        <?php echo date('d.m.Y', strtotime($product['created_at'])); ?>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    <?php endwhile; ?>
                </div>
                
                <!-- Пагинация -->
                <?php if ($total_pages > 1): ?>
                <nav aria-label="Навигация по страницам" class="mt-5">
                    <ul class="pagination justify-content-center">
                        <?php if ($page > 1): ?>
                        <li class="page-item">
                            <a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['page' => $page - 1])); ?>">
                                <i class="fas fa-chevron-left"></i>
                            </a>
                        </li>
                        <?php endif; ?>
                        
                        <?php
                        $start = max(1, $page - 2);
                        $end = min($total_pages, $page + 2);
                        
                        if ($start > 1) {
                            echo '<li class="page-item"><a class="page-link" href="?' . http_build_query(array_merge($_GET, ['page' => 1])) . '">1</a></li>';
                            if ($start > 2) {
                                echo '<li class="page-item disabled"><span class="page-link">...</span></li>';
                            }
                        }
                        
                        for ($i = $start; $i <= $end; $i++):
                        ?>
                        <li class="page-item <?php echo $i == $page ? 'active' : ''; ?>">
                            <a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['page' => $i])); ?>">
                                <?php echo $i; ?>
                            </a>
                        </li>
                        <?php endfor; ?>
                        
                        if ($end < $total_pages) {
                            if ($end < $total_pages - 1) {
                                echo '<li class="page-item disabled"><span class="page-link">...</span></li>';
                            }
                            echo '<li class="page-item"><a class="page-link" href="?' . http_build_query(array_merge($_GET, ['page' => $total_pages])) . '">' . $total_pages . '</a></li>';
                        }
                        ?>
                        
                        <?php if ($page < $total_pages): ?>
                        <li class="page-item">
                            <a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['page' => $page + 1])); ?>">
                                <i class="fas fa-chevron-right"></i>
                            </a>
                        </li>
                        <?php endif; ?>
                    </ul>
                </nav>
                <?php endif; ?>
                
            <?php else: ?>
                <div class="card">
                    <div class="card-body text-center py-5">
                        <i class="fas fa-search fa-4x text-muted mb-4"></i>
                        <h3 class="text-muted mb-3">Товары не найдены</h3>
                        <p class="text-muted mb-4">
                            Попробуйте изменить параметры фильтрации или воспользуйтесь поиском
                        </p>
                        <a href="/catalog.php" class="btn btn-primary">
                            <i class="fas fa-redo me-2"></i>Показать все товары
                        </a>
                    </div>
                </div>
            <?php endif; ?>
            
            <!-- SEO текст -->
            <div class="card mt-5">
                <div class="card-body">
                    <h5 class="card-title">Каталог цифровых товаров</h5>
                    <p class="card-text">
                        В нашем каталоге представлен широкий выбор цифровых товаров: программы, игры, шаблоны, курсы и графика. 
                        Все товары доступны для мгновенного скачивания после оплаты. Мы регулярно обновляем ассортимент и 
                        предлагаем только качественный контент от проверенных авторов.
                    </p>
                    <p class="card-text">
                        Используйте фильтры для быстрого поиска нужного товара. Все покупки защищены гарантией возврата 
                        средств в течение 14 дней. Если у вас возникли вопросы - обратитесь в нашу службу поддержки.
                    </p>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    // Переключение вида отображения
    document.querySelectorAll('input[name="viewMode"]').forEach(radio => {
        radio.addEventListener('change', function() {
            if (this.id === 'viewGrid') {
                document.getElementById('productsGrid').classList.remove('d-none');
                document.getElementById('productsList').classList.add('d-none');
            } else {
                document.getElementById('productsGrid').classList.add('d-none');
                document.getElementById('productsList').classList.remove('d-none');
            }
        });
    });
    
    // Слайдер цены
    const priceRangeMin = document.getElementById('priceRangeMin');
    const priceRangeMax = document.getElementById('priceRangeMax');
    const minPriceInput = document.getElementById('minPrice');
    const maxPriceInput = document.getElementById('maxPrice');
    const priceMinLabel = document.getElementById('priceMinLabel');
    const priceMaxLabel = document.getElementById('priceMaxLabel');
    
    function updatePriceLabels() {
        const min = parseInt(priceRangeMin.value);
        const max = parseInt(priceRangeMax.value);
        
        priceMinLabel.textContent = min.toLocaleString('ru-RU') + ' ₽';
        priceMaxLabel.textContent = max.toLocaleString('ru-RU') + ' ₽';
        
        minPriceInput.value = min;
        maxPriceInput.value = max;
        
        // Ограничение минимального значения
        if (priceRangeMin.value > priceRangeMax.value) {
            priceRangeMin.value = priceRangeMax.value;
        }
    }
    
    if (priceRangeMin && priceRangeMax) {
        priceRangeMin.addEventListener('input', updatePriceLabels);
        priceRangeMax.addEventListener('input', updatePriceLabels);
        
        // Синхронизация с полями ввода
        minPriceInput.addEventListener('input', function() {
            const value = parseInt(this.value) || 0;
            priceRangeMin.value = Math.min(value, parseInt(priceRangeMax.value));
            updatePriceLabels();
        });
        
        maxPriceInput.addEventListener('input', function() {
            const value = parseInt(this.value) || 0;
            const maxRange = parseInt(priceRangeMax.max);
            priceRangeMax.value = Math.min(Math.max(value, parseInt(priceRangeMin.value)), maxRange);
            updatePriceLabels();
        });
        
        updatePriceLabels();
    }
    
    // Применение фильтров
    const applyFiltersBtn = document.getElementById('applyFilters');
    const sortSelect = document.getElementById('sortSelect');
    
    if (applyFiltersBtn) {
        applyFiltersBtn.addEventListener('click', function() {
            const params = new URLSearchParams(window.location.search);
            
            // Категория
            const category = <?php echo $category_id; ?>;
            if (category > 0) {
                params.set('category', category);
            }
            
            // Цена
            const minPrice = parseInt(minPriceInput.value) || 0;
            const maxPrice = parseInt(maxPriceInput.value) || '';
            
            if (minPrice > 0) {
                params.set('min_price', minPrice);
            } else {
                params.delete('min_price');
            }
            
            if (maxPrice && maxPrice > 0) {
                params.set('max_price', maxPrice);
            } else {
                params.delete('max_price');
            }
            
            // Сортировка
            if (sortSelect.value !== 'newest') {
                params.set('sort', sortSelect.value);
            } else {
                params.delete('sort');
            }
            
            // Поиск
            const search = '<?php echo $search; ?>';
            if (search) {
                params.set('search', search);
            }
            
            // Переход с обновленными параметрами
            window.location.href = '/catalog.php?' + params.toString();
        });
    }
    
    // Быстрая загрузка товаров с AJAX (для пагинации без перезагрузки)
    document.querySelectorAll('.pagination a').forEach(link => {
        link.addEventListener('click', function(e) {
            if (window.innerWidth > 768) { // Только на десктопе
                e.preventDefault();
                
                const url = this.getAttribute('href');
                const container = document.getElementById('productsGrid');
                
                if (container) {
                    // Показываем индикатор загрузки
                    const loadingHtml = `
                        <div class="col-12 text-center py-5">
                            <div class="loading" style="width: 50px; height: 50px;"></div>
                            <p class="mt-3 text-muted">Загрузка товаров...</p>
                        </div>
                    `;
                    container.innerHTML = loadingHtml;
                    
                    // Загружаем новую страницу
                    fetch(url)
                    .then(response => response.text())
                    .then(html => {
                        // Парсим HTML и извлекаем товары
                        const parser = new DOMParser();
                        const doc = parser.parseFromString(html, 'text/html');
                        const newProducts = doc.getElementById('productsGrid').innerHTML;
                        const newPagination = doc.querySelector('.pagination')?.outerHTML || '';
                        
                        // Обновляем товары
                        container.innerHTML = newProducts;
                        
                        // Обновляем пагинацию
                        const paginationContainer = document.querySelector('.pagination');
                        if (paginationContainer && newPagination) {
                            paginationContainer.outerHTML = newPagination;
                        }
                        
                        // Обновляем URL в браузере без перезагрузки
                        window.history.pushState({}, '', url);
                        
                        // Прокручиваем к началу товаров
                        container.scrollIntoView({ behavior: 'smooth' });
                        
                        // Инициализируем обработчики событий для новых товаров
                        initProductCards();
                    })
                    .catch(error => {
                        console.error('Ошибка загрузки:', error);
                        container.innerHTML = '<div class="col-12 text-center py-5 text-danger">Ошибка загрузки товаров</div>';
                    });
                }
            }
        });
    });
    
    // Обработка нажатия кнопки "Назад" в браузере
    window.addEventListener('popstate', function() {
        location.reload();
    });
    
    function initProductCards() {
        // Инициализация кнопок "В корзину" для новых товаров
        document.querySelectorAll('.add-to-cart').forEach(button => {
            button.addEventListener('click', function() {
                const productId = this.dataset.productId;
                addToCart(productId);
            });
        });
    }
    
    // Инициализация при загрузке
    initProductCards();
});

function addToCart(productId) {
    fetch('/api/cart.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'X-CSRF-Token': window.App.csrfToken
        },
        body: JSON.stringify({
            action: 'add',
            product_id: productId
        })
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showNotification('Товар добавлен в корзину!', 'success');
            
            // Обновляем счетчик в шапке
            const cartCount = document.getElementById('cart-count');
            if (cartCount) {
                cartCount.textContent = data.cart_count;
            } else if (data.cart_count > 0) {
                // Создаем бейдж если его нет
                const cartIcon = document.querySelector('.fa-shopping-cart').closest('a');
                const badge = document.createElement('span');
                badge.className = 'cart-badge';
                badge.id = 'cart-count';
                badge.textContent = data.cart_count;
                cartIcon.appendChild(badge);
            }
        } else {
            showNotification(data.error, 'error');
        }
    });
}
</script>

<?php include 'templates/footer.php'; ?>