<?php
require_once 'includes/config.php';
require_once 'includes/functions.php';

$title = "Категории товаров";
$theme = isset($_COOKIE['theme']) ? $_COOKIE['theme'] : 'light';

$db = db();

// Получаем категории с количеством товаров
$categories = $db->query("
    SELECT 
        c.*, 
        COUNT(p.id) as product_count,
        (SELECT COUNT(*) FROM products p2 
         WHERE p2.category_id = c.id 
         AND p2.discount_price IS NOT NULL 
         AND p2.status = 'active') as discount_count
    FROM categories c
    LEFT JOIN products p ON c.id = p.category_id AND p.status = 'active'
    WHERE c.is_active = 1
    GROUP BY c.id
    ORDER BY c.sort_order, c.name
");

// Получаем популярные товары из каждой категории
$popular_by_category = [];
$cat_result = $db->query("
    SELECT c.id as category_id, c.name as category_name,
           p.id, p.name, p.price, p.discount_price, p.image, p.downloads_count
    FROM categories c
    LEFT JOIN products p ON (
        p.id = (
            SELECT p2.id 
            FROM products p2 
            WHERE p2.category_id = c.id 
            AND p2.status = 'active'
            ORDER BY p2.downloads_count DESC 
            LIMIT 1
        )
    )
    WHERE c.is_active = 1
    ORDER BY c.sort_order
");

while ($row = $cat_result->fetch_assoc()) {
    if ($row['id']) {
        $popular_by_category[$row['category_id']] = $row;
    }
}

include 'templates/header.php';
?>

<div class="container py-5">
    <div class="row mb-5">
        <div class="col-12">
            <h1 class="display-5 mb-3">Категории товаров</h1>
            <p class="lead text-muted">
                Выберите интересующую вас категорию цифровых товаров. 
                У нас представлен широкий ассортимент программ, игр, курсов и другого цифрового контента.
            </p>
        </div>
    </div>
    
    <?php if ($categories->num_rows > 0): ?>
        <div class="row row-cols-1 row-cols-md-2 row-cols-lg-3 g-4 mb-5">
            <?php while ($category = $categories->fetch_assoc()): 
                $has_popular = isset($popular_by_category[$category['id']]);
                $popular_item = $has_popular ? $popular_by_category[$category['id']] : null;
            ?>
            <div class="col">
                <div class="card category-card h-100 shadow-sm hover-shadow">
                    <div class="card-body">
                        <div class="d-flex align-items-start mb-3">
                            <div class="flex-shrink-0">
                                <div class="category-icon rounded-circle bg-primary bg-opacity-10 p-3">
                                    <i class="fas fa-<?php 
                                        $icons = ['programs' => 'code', 'games' => 'gamepad', 'templates' => 'file-alt', 
                                                 'courses' => 'graduation-cap', 'graphics' => 'palette', 'music' => 'music',
                                                 'books' => 'book', 'photos' => 'camera'];
                                        echo $icons[$category['slug']] ?? 'folder';
                                    ?> fa-2x text-primary"></i>
                                </div>
                            </div>
                            <div class="flex-grow-1 ms-3">
                                <h5 class="card-title mb-1">
                                    <a href="/catalog.php?category=<?php echo $category['id']; ?>" class="text-decoration-none">
                                        <?php echo htmlspecialchars($category['name']); ?>
                                    </a>
                                </h5>
                                <p class="card-text text-muted small">
                                    <?php echo htmlspecialchars($category['description']); ?>
                                </p>
                            </div>
                        </div>
                        
                        <div class="d-flex justify-content-between align-items-center mb-3">
                            <div>
                                <span class="badge bg-primary rounded-pill">
                                    <i class="fas fa-box me-1"></i>
                                    <?php echo $category['product_count']; ?> товаров
                                </span>
                                
                                <?php if ($category['discount_count'] > 0): ?>
                                <span class="badge bg-success rounded-pill ms-2">
                                    <i class="fas fa-percent me-1"></i>
                                    <?php echo $category['discount_count']; ?> со скидкой
                                </span>
                                <?php endif; ?>
                            </div>
                            
                            <a href="/catalog.php?category=<?php echo $category['id']; ?>" class="btn btn-sm btn-outline-primary">
                                Смотреть все
                            </a>
                        </div>
                        
                        <?php if ($has_popular && $popular_item): ?>
                        <div class="popular-product border-top pt-3">
                            <h6 class="small text-muted mb-2">
                                <i class="fas fa-fire text-warning me-1"></i>Популярный товар
                            </h6>
                            <div class="d-flex align-items-center">
                                <?php if ($popular_item['image']): ?>
                                <div class="flex-shrink-0 me-3">
                                    <img src="<?php echo $popular_item['image']; ?>" 
                                         alt="<?php echo htmlspecialchars($popular_item['name']); ?>" 
                                         class="rounded" 
                                         style="width: 60px; height: 60px; object-fit: cover;">
                                </div>
                                <?php endif; ?>
                                <div class="flex-grow-1">
                                    <h6 class="mb-1">
                                        <a href="/product.php?id=<?php echo $popular_item['id']; ?>" class="text-decoration-none small">
                                            <?php echo htmlspecialchars(mb_substr($popular_item['name'], 0, 40)) . (mb_strlen($popular_item['name']) > 40 ? '...' : ''); ?>
                                        </a>
                                    </h6>
                                    <div class="d-flex justify-content-between align-items-center">
                                        <div class="price small">
                                            <?php if ($popular_item['discount_price']): ?>
                                                <span class="text-muted text-decoration-line-through me-2">
                                                    <?php echo number_format($popular_item['price'], 0); ?> ₽
                                                </span>
                                                <span class="fw-bold text-success">
                                                    <?php echo number_format($popular_item['discount_price'], 0); ?> ₽
                                                </span>
                                            <?php else: ?>
                                                <span class="fw-bold text-primary">
                                                    <?php echo number_format($popular_item['price'], 0); ?> ₽
                                                </span>
                                            <?php endif; ?>
                                        </div>
                                        <div class="text-muted extra-small">
                                            <i class="fas fa-download me-1"></i>
                                            <?php echo number_format($popular_item['downloads_count']); ?>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
            <?php endwhile; ?>
        </div>
    <?php else: ?>
        <div class="card">
            <div class="card-body text-center py-5">
                <i class="fas fa-tags fa-4x text-muted mb-4"></i>
                <h3 class="text-muted mb-3">Категории пока не добавлены</h3>
                <p class="text-muted mb-4">
                    В настоящее время категории товаров находятся в разработке
                </p>
                <a href="/catalog.php" class="btn btn-primary">
                    <i class="fas fa-shopping-bag me-2"></i>Перейти в каталог
                </a>
            </div>
        </div>
    <?php endif; ?>
    
    <!-- Рекомендации -->
    <div class="row mt-5">
        <div class="col-12">
            <div class="card bg-light">
                <div class="card-body">
                    <div class="row align-items-center">
                        <div class="col-md-8">
                            <h5 class="card-title">Не нашли нужную категорию?</h5>
                            <p class="card-text">
                                Отправьте нам запрос на добавление новой категории товаров. 
                                Мы постоянно расширяем ассортимент и учитываем пожелания наших покупателей.
                            </p>
                        </div>
                        <div class="col-md-4 text-md-end">
                            <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#requestCategoryModal">
                                <i class="fas fa-plus-circle me-2"></i>Запросить категорию
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <!-- SEO текст -->
    <div class="row mt-5">
        <div class="col-12">
            <div class="card">
                <div class="card-body">
                    <h5 class="card-title">Все категории цифровых товаров в одном месте</h5>
                    <p class="card-text">
                        Наш магазин предлагает широкий выбор цифровых товаров, организованных по категориям для вашего удобства. 
                        Каждая категория содержит тщательно отобранные товары от проверенных авторов и разработчиков.
                    </p>
                    <p class="card-text">
                        <strong>Программы и софт</strong> - полезные утилиты, профессиональные программы, инструменты для работы и развлечений.<br>
                        <strong>Игры</strong> - увлекательные игры для всех платформ, от классических головоломок до современных экшенов.<br>
                        <strong>Шаблоны</strong> - готовые решения для сайтов, презентаций, документов и дизайна.<br>
                        <strong>Курсы и обучение</strong> - образовательные материалы, видеоуроки, онлайн-курсы по различным темам.<br>
                        <strong>Графика и дизайн</strong> - стоковые фото, иконки, шрифты, текстуры и другие графические материалы.
                    </p>
                    <p class="card-text">
                        Мы постоянно обновляем ассортимент и добавляем новые категории в соответствии с запросами наших клиентов. 
                        Все товары проходят проверку качества и доступны для мгновенной загрузки после оплаты.
                    </p>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Модальное окно запроса категории -->
<div class="modal fade" id="requestCategoryModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Запрос новой категории</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form id="requestCategoryForm">
                <div class="modal-body">
                    <div class="mb-3">
                        <label for="categoryName" class="form-label">Название категории *</label>
                        <input type="text" class="form-control" id="categoryName" required>
                    </div>
                    <div class="mb-3">
                        <label for="categoryDescription" class="form-label">Описание категории</label>
                        <textarea class="form-control" id="categoryDescription" rows="3"></textarea>
                        <div class="form-text">Опишите, какие товары должны быть в этой категории</div>
                    </div>
                    <div class="mb-3">
                        <label for="userEmail" class="form-label">Ваш email *</label>
                        <input type="email" class="form-control" id="userEmail" required>
                        <div class="form-text">На этот email мы отправим ответ</div>
                    </div>
                    <div class="form-check mb-3">
                        <input type="checkbox" class="form-check-input" id="subscribeNews" checked>
                        <label class="form-check-label" for="subscribeNews">
                            Подписаться на новости о новых категориях
                        </label>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Отмена</button>
                    <button type="submit" class="btn btn-primary">Отправить запрос</button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    // Анимация карточек категорий
    const categoryCards = document.querySelectorAll('.category-card');
    categoryCards.forEach(card => {
        card.addEventListener('mouseenter', function() {
            this.style.transform = 'translateY(-5px)';
            this.style.transition = 'transform 0.3s, box-shadow 0.3s';
        });
        
        card.addEventListener('mouseleave', function() {
            this.style.transform = 'translateY(0)';
        });
    });
    
    // Форма запроса категории
    const requestForm = document.getElementById('requestCategoryForm');
    if (requestForm) {
        requestForm.addEventListener('submit', function(e) {
            e.preventDefault();
            
            const formData = {
                name: document.getElementById('categoryName').value,
                description: document.getElementById('categoryDescription').value,
                email: document.getElementById('userEmail').value,
                subscribe: document.getElementById('subscribeNews').checked
            };
            
            fetch('/api/request-category.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-Token': window.App.csrfToken
                },
                body: JSON.stringify(formData)
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    showNotification('Запрос отправлен успешно! Спасибо за ваше предложение.', 'success');
                    const modal = bootstrap.Modal.getInstance(document.getElementById('requestCategoryModal'));
                    modal.hide();
                    requestForm.reset();
                } else {
                    showNotification(data.error || 'Ошибка отправки запроса', 'error');
                }
            });
        });
    }
});
</script>

<style>
.category-card {
    transition: all 0.3s ease;
    border: 1px solid rgba(0,0,0,0.1);
}

.category-card:hover {
    border-color: var(--primary-color);
    box-shadow: 0 10px 20px rgba(0,0,0,0.1) !important;
}

.category-icon {
    width: 70px;
    height: 70px;
    display: flex;
    align-items: center;
    justify-content: center;
}

.hover-shadow:hover {
    box-shadow: 0 5px 15px rgba(0,0,0,0.1);
}

.popular-product {
    background-color: rgba(var(--primary-color-rgb, 13, 110, 253), 0.05);
    border-radius: 8px;
    margin-top: 10px;
}
</style>

<?php include 'templates/footer.php'; ?>