<?php
require_once 'includes/config.php';
require_once 'includes/functions.php';

if (!isLoggedIn()) {
    header('Location: /login.php?redirect=checkout');
    exit;
}

$title = "Оформление заказа";
$theme = isset($_COOKIE['theme']) ? $_COOKIE['theme'] : 'light';

// Получаем товары из корзины
$db = db();
$userId = $_SESSION['user_id'];

$cartItems = $db->query("
    SELECT c.*, p.name, p.price, p.discount_price, p.image
    FROM cart c
    JOIN products p ON c.product_id = p.id
    WHERE c.user_id = $userId AND p.status = 'active'
");

$total = 0;
$items = [];

while ($item = $cartItems->fetch_assoc()) {
    $price = $item['discount_price'] ?: $item['price'];
    $items[] = $item;
    $total += $price * $item['quantity'];
}

// Обработка оформления заказа
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['payment_method'])) {
    $paymentMethod = sanitize($_POST['payment_method']);
    
    // Генерируем номер заказа
    $orderNumber = 'ORD-' . date('Ymd') . '-' . strtoupper(bin2hex(random_bytes(3)));
    
    $db->begin_transaction();
    
    try {
        // Создаем заказ
        $stmt = $db->prepare("
            INSERT INTO orders (user_id, order_number, total_amount, payment_method, created_at)
            VALUES (?, ?, ?, ?, NOW())
        ");
        $stmt->bind_param("isds", $userId, $orderNumber, $total, $paymentMethod);
        $stmt->execute();
        
        $orderId = $db->insert_id;
        
        // Добавляем товары в заказ
        foreach ($items as $item) {
            $price = $item['discount_price'] ?: $item['price'];
            
            $stmt = $db->prepare("
                INSERT INTO order_items (order_id, product_id, price)
                VALUES (?, ?, ?)
            ");
            $stmt->bind_param("iid", $orderId, $item['product_id'], $price);
            $stmt->execute();
        }
        
        // Очищаем корзину
        $stmt = $db->prepare("DELETE FROM cart WHERE user_id = ?");
        $stmt->bind_param("i", $userId);
        $stmt->execute();
        
        $db->commit();
        
        // Перенаправляем на оплату
        if ($paymentMethod === 'yoomoney') {
            header("Location: /payment/yoomoney.php?order_id=" . $orderNumber);
        } elseif ($paymentMethod === 'azvox') {
            header("Location: /payment/azvox.php?order_id=" . $orderNumber);
        } else {
            header("Location: /checkout-success.php?order=" . $orderNumber);
        }
        exit;
        
    } catch (Exception $e) {
        $db->rollback();
        $error = "Ошибка при создании заказа: " . $e->getMessage();
    }
}

include 'templates/header.php';
?>

<div class="container py-5">
    <div class="row">
        <div class="col-lg-8">
            <h2 class="mb-4">Оформление заказа</h2>
            
            <?php if (!empty($error)): ?>
                <div class="alert alert-danger"><?php echo $error; ?></div>
            <?php endif; ?>
            
            <?php if (empty($items)): ?>
                <div class="alert alert-info">
                    Ваша корзина пуста. <a href="/catalog.php">Перейти в каталог</a>
                </div>
            <?php else: ?>
                <div class="card mb-4">
                    <div class="card-header">
                        <h5>Товары в заказе</h5>
                    </div>
                    <div class="card-body">
                        <?php foreach ($items as $item): ?>
                            <div class="d-flex justify-content-between align-items-center mb-3 pb-3 border-bottom">
                                <div class="d-flex align-items-center">
                                    <?php if ($item['image']): ?>
                                        <img src="<?php echo $item['image']; ?>" 
                                             alt="<?php echo htmlspecialchars($item['name']); ?>"
                                             width="60" height="60" class="rounded mr-3">
                                    <?php endif; ?>
                                    <div>
                                        <h6 class="mb-0"><?php echo htmlspecialchars($item['name']); ?></h6>
                                        <small class="text-muted">Количество: <?php echo $item['quantity']; ?></small>
                                    </div>
                                </div>
                                <div class="text-right">
                                    <span class="h6 mb-0">
                                        <?php 
                                        $price = $item['discount_price'] ?: $item['price'];
                                        echo number_format($price * $item['quantity'], 2); ?> ₽
                                    </span>
                                </div>
                            </div>
                        <?php endforeach; ?>
                        
                        <div class="d-flex justify-content-between align-items-center mt-3">
                            <span class="h5 mb-0">Итого:</span>
                            <span class="h4 text-primary"><?php echo number_format($total, 2); ?> ₽</span>
                        </div>
                    </div>
                </div>
                
                <form method="POST" id="checkout-form">
                    <div class="card mb-4">
                        <div class="card-header">
                            <h5>Способ оплаты</h5>
                        </div>
                        <div class="card-body">
                            <div class="form-group">
                                <div class="custom-control custom-radio mb-2">
                                    <input type="radio" id="yoomoney" name="payment_method" value="yoomoney" class="custom-control-input" required>
                                    <label class="custom-control-label" for="yoomoney">
                                        <i class="fab fa-yandex-money fa-lg mr-2"></i>
                                        ЮMoney (Яндекс.Деньги)
                                    </label>
                                </div>
                                <div class="custom-control custom-radio mb-2">
                                    <input type="radio" id="azvox" name="payment_method" value="azvox" class="custom-control-input">
                                    <label class="custom-control-label" for="azvox">
                                        <i class="fas fa-credit-card fa-lg mr-2"></i>
                                        Azvox (банковские карты)
                                    </label>
                                </div>
                                <?php if ($_SESSION['user_balance'] >= $total): ?>
                                <div class="custom-control custom-radio">
                                    <input type="radio" id="balance" name="payment_method" value="balance" class="custom-control-input">
                                    <label class="custom-control-label" for="balance">
                                        <i class="fas fa-wallet fa-lg mr-2"></i>
                                        Баланс (<?php echo number_format($_SESSION['user_balance'], 2); ?> ₽)
                                    </label>
                                </div>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <button type="submit" class="btn btn-primary btn-lg btn-block">
                            <i class="fas fa-lock mr-2"></i>Перейти к оплате
                        </button>
                    </div>
                    
                    <div class="alert alert-info small">
                        <i class="fas fa-info-circle mr-2"></i>
                        После оплаты вы получите ссылку для скачивания на указанный email.
                        Ссылка действительна в течение 7 дней и для 3 скачиваний.
                    </div>
                </form>
            <?php endif; ?>
        </div>
        
        <div class="col-lg-4">
            <div class="card">
                <div class="card-header">
                    <h5>Информация о покупателе</h5>
                </div>
                <div class="card-body">
                    <p><strong>Имя:</strong> <?php echo htmlspecialchars($_SESSION['user_name']); ?></p>
                    <p><strong>Email:</strong> <?php echo htmlspecialchars($_SESSION['user_email']); ?></p>
                    <p><strong>Баланс:</strong> <?php echo number_format($_SESSION['user_balance'] ?? 0, 2); ?> ₽</p>
                    
                    <hr>
                    
                    <h6>Поддержка</h6>
                    <p class="small text-muted">
                        Если возникнут проблемы с оплатой или скачиванием, 
                        обратитесь в службу поддержки: support@<?php echo parse_url(SITE_URL, PHP_URL_HOST); ?>
                    </p>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
document.getElementById('checkout-form').addEventListener('submit', function(e) {
    const paymentMethod = document.querySelector('input[name="payment_method"]:checked');
    if (!paymentMethod) {
        e.preventDefault();
        alert('Пожалуйста, выберите способ оплаты');
        return false;
    }
    
    // Показываем индикатор загрузки
    const submitBtn = this.querySelector('button[type="submit"]');
    submitBtn.innerHTML = '<span class="loading"></span> Обработка...';
    submitBtn.disabled = true;
});
</script>

<?php include 'templates/footer.php'; ?>