<?php
require_once 'includes/config.php';
require_once 'includes/functions.php';

$title = "Часто задаваемые вопросы (FAQ)";
$theme = isset($_COOKIE['theme']) ? $_COOKIE['theme'] : 'light';

$db = db();

// Получаем FAQ из базы данных
$faq_categories = $db->query("
    SELECT fc.*, 
           COUNT(f.id) as question_count
    FROM faq_categories fc
    LEFT JOIN faq f ON fc.id = f.category_id AND f.is_active = 1
    WHERE fc.is_active = 1
    GROUP BY fc.id
    ORDER BY fc.sort_order
");

// Получаем все активные вопросы
$faq_questions = $db->query("
    SELECT f.*, fc.name as category_name, fc.slug as category_slug
    FROM faq f
    LEFT JOIN faq_categories fc ON f.category_id = fc.id
    WHERE f.is_active = 1
    ORDER BY f.sort_order, f.created_at DESC
");

include 'templates/header.php';
?>

<div class="container py-5">
    <!-- Заголовок -->
    <div class="row mb-5">
        <div class="col-12">
            <h1 class="display-5 mb-3">
                <i class="fas fa-question-circle text-primary me-2"></i>Часто задаваемые вопросы
            </h1>
            <p class="lead text-muted">
                Ответы на самые популярные вопросы о покупке, скачивании и использовании цифровых товаров.
            </p>
            
            <!-- Быстрый поиск -->
            <div class="row mt-4">
                <div class="col-lg-8 mx-auto">
                    <div class="card">
                        <div class="card-body">
                            <div class="input-group">
                                <input type="text" 
                                       class="form-control" 
                                       id="faqSearch" 
                                       placeholder="Введите вопрос или ключевое слово...">
                                <button class="btn btn-primary" type="button" id="searchFaqBtn">
                                    <i class="fas fa-search"></i>
                                </button>
                            </div>
                            <div class="mt-2 text-muted small">
                                Например: "как скачать", "возврат средств", "оплата"
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Категории FAQ -->
    <?php if ($faq_categories->num_rows > 0): ?>
    <div class="row mb-5">
        <div class="col-12">
            <h3 class="mb-4">Выберите категорию вопросов</h3>
            
            <div class="row row-cols-1 row-cols-md-2 row-cols-lg-3 g-4">
                <?php while ($category = $faq_categories->fetch_assoc()): ?>
                <div class="col">
                    <a href="#category-<?php echo $category['slug']; ?>" 
                       class="card text-decoration-none faq-category-card h-100">
                        <div class="card-body text-center">
                            <div class="category-icon mb-3">
                                <i class="fas fa-<?php 
                                    $icons = [
                                        'buying' => 'shopping-cart',
                                        'download' => 'download',
                                        'payment' => 'credit-card',
                                        'account' => 'user',
                                        'technical' => 'cog',
                                        'refund' => 'undo',
                                        'security' => 'shield-alt',
                                        'general' => 'info-circle'
                                    ];
                                    echo $icons[$category['slug']] ?? 'question-circle';
                                ?> fa-3x text-primary"></i>
                            </div>
                            <h5 class="card-title"><?php echo htmlspecialchars($category['name']); ?></h5>
                            <p class="card-text text-muted small">
                                <?php echo htmlspecialchars($category['description']); ?>
                            </p>
                            <div class="badge bg-primary rounded-pill">
                                <?php echo $category['question_count']; ?> вопросов
                            </div>
                        </div>
                    </a>
                </div>
                <?php endwhile; ?>
            </div>
        </div>
    </div>
    <?php endif; ?>
    
    <!-- FAQ по категориям -->
    <div class="row">
        <div class="col-12">
            <?php 
            if ($faq_categories->num_rows > 0) {
                $faq_categories->data_seek(0); // Сбрасываем указатель
                
                while ($category = $faq_categories->fetch_assoc()):
                    // Получаем вопросы для этой категории
                    $cat_questions = $db->query("
                        SELECT * FROM faq 
                        WHERE category_id = {$category['id']} 
                        AND is_active = 1 
                        ORDER BY sort_order, created_at DESC
                    ");
                    
                    if ($cat_questions->num_rows > 0):
            ?>
            <div class="faq-category mb-5" id="category-<?php echo $category['slug']; ?>">
                <h3 class="border-bottom pb-3 mb-4">
                    <i class="fas fa-<?php 
                        $icons = [
                            'buying' => 'shopping-cart',
                            'download' => 'download',
                            'payment' => 'credit-card',
                            'account' => 'user',
                            'technical' => 'cog',
                            'refund' => 'undo',
                            'security' => 'shield-alt',
                            'general' => 'info-circle'
                        ];
                        echo $icons[$category['slug']] ?? 'question-circle';
                    ?> me-2 text-primary"></i>
                    <?php echo htmlspecialchars($category['name']); ?>
                </h3>
                
                <div class="accordion" id="accordion-<?php echo $category['slug']; ?>">
                    <?php 
                    $question_num = 1;
                    while ($question = $cat_questions->fetch_assoc()): 
                        $question_id = "question-{$category['slug']}-{$question['id']}";
                    ?>
                    <div class="accordion-item">
                        <h3 class="accordion-header" id="heading-<?php echo $question_id; ?>">
                            <button class="accordion-button <?php echo $question_num > 1 ? 'collapsed' : ''; ?>" 
                                    type="button" 
                                    data-bs-toggle="collapse" 
                                    data-bs-target="#collapse-<?php echo $question_id; ?>" 
                                    aria-expanded="<?php echo $question_num == 1 ? 'true' : 'false'; ?>"
                                    aria-controls="collapse-<?php echo $question_id; ?>">
                                <span class="question-number me-3"><?php echo $question_num; ?>.</span>
                                <?php echo htmlspecialchars($question['question']); ?>
                            </button>
                        </h3>
                        <div id="collapse-<?php echo $question_id; ?>" 
                             class="accordion-collapse collapse <?php echo $question_num == 1 ? 'show' : ''; ?>" 
                             aria-labelledby="heading-<?php echo $question_id; ?>"
                             data-bs-parent="#accordion-<?php echo $category['slug']; ?>">
                            <div class="accordion-body">
                                <div class="faq-answer">
                                    <?php echo nl2br(htmlspecialchars($question['answer'])); ?>
                                    
                                    <?php if (!empty($question['additional_info'])): ?>
                                    <div class="mt-3 p-3 bg-light rounded">
                                        <strong>Дополнительная информация:</strong><br>
                                        <?php echo nl2br(htmlspecialchars($question['additional_info'])); ?>
                                    </div>
                                    <?php endif; ?>
                                    
                                    <?php if (!empty($question['related_links'])): ?>
                                    <div class="mt-3">
                                        <strong>Полезные ссылки:</strong>
                                        <ul class="mb-0">
                                            <?php 
                                            $links = explode("\n", $question['related_links']);
                                            foreach ($links as $link):
                                                if (trim($link)):
                                            ?>
                                            <li><a href="<?php echo htmlspecialchars(trim($link)); ?>" target="_blank"><?php echo htmlspecialchars(trim($link)); ?></a></li>
                                            <?php 
                                                endif;
                                            endforeach; 
                                            ?>
                                        </ul>
                                    </div>
                                    <?php endif; ?>
                                    
                                    <div class="mt-3 text-muted small">
                                        <i class="fas fa-calendar me-1"></i>
                                        Обновлено: <?php echo date('d.m.Y', strtotime($question['updated_at'] ?: $question['created_at'])); ?>
                                        <?php if ($question['views_count'] > 0): ?>
                                        <span class="ms-3">
                                            <i class="fas fa-eye me-1"></i>
                                            Просмотров: <?php echo number_format($question['views_count']); ?>
                                        </span>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    <?php 
                    $question_num++;
                    endwhile; 
                    ?>
                </div>
            </div>
            <?php 
                    endif;
                endwhile;
            } else {
                // Если нет категорий, показываем все вопросы
                if ($faq_questions->num_rows > 0):
            ?>
            <div class="accordion" id="accordionAll">
                <?php 
                $question_num = 1;
                while ($question = $faq_questions->fetch_assoc()): 
                    $question_id = "question-all-{$question['id']}";
                ?>
                <div class="accordion-item">
                    <h3 class="accordion-header" id="heading-<?php echo $question_id; ?>">
                        <button class="accordion-button <?php echo $question_num > 1 ? 'collapsed' : ''; ?>" 
                                type="button" 
                                data-bs-toggle="collapse" 
                                data-bs-target="#collapse-<?php echo $question_id; ?>" 
                                aria-expanded="<?php echo $question_num == 1 ? 'true' : 'false'; ?>"
                                aria-controls="collapse-<?php echo $question_id; ?>">
                            <span class="badge bg-secondary me-2"><?php echo htmlspecialchars($question['category_name']); ?></span>
                            <?php echo htmlspecialchars($question['question']); ?>
                        </button>
                    </h3>
                    <div id="collapse-<?php echo $question_id; ?>" 
                         class="accordion-collapse collapse <?php echo $question_num == 1 ? 'show' : ''; ?>" 
                         aria-labelledby="heading-<?php echo $question_id; ?>"
                         data-bs-parent="#accordionAll">
                        <div class="accordion-body">
                            <?php echo nl2br(htmlspecialchars($question['answer'])); ?>
                        </div>
                    </div>
                </div>
                <?php 
                $question_num++;
                endwhile; 
                ?>
            </div>
            <?php 
                else:
            ?>
            <div class="card">
                <div class="card-body text-center py-5">
                    <i class="fas fa-question-circle fa-4x text-muted mb-4"></i>
                    <h3 class="text-muted mb-3">Вопросы и ответы не найдены</h3>
                    <p class="text-muted mb-4">
                        В настоящее время раздел FAQ находится в разработке
                    </p>
                </div>
            </div>
            <?php 
                endif;
            }
            ?>
        </div>
    </div>
    
    <!-- Не нашли ответ? -->
    <div class="row mt-5">
        <div class="col-12">
            <div class="card bg-light">
                <div class="card-body">
                    <div class="row align-items-center">
                        <div class="col-md-8">
                            <h5 class="card-title mb-3">
                                <i class="fas fa-comment-dots text-primary me-2"></i>Не нашли ответ на свой вопрос?
                            </h5>
                            <p class="card-text mb-0">
                                Задайте свой вопрос нашей службе поддержки. Мы ответим в течение 24 часов 
                                и добавим ваш вопрос в этот раздел, если он окажется полезным для других пользователей.
                            </p>
                        </div>
                        <div class="col-md-4 text-md-end">
                            <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#askQuestionModal">
                                <i class="fas fa-question me-2"></i>Задать вопрос
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Полезные ссылки -->
    <div class="row mt-5">
        <div class="col-12">
            <div class="card">
                <div class="card-body">
                    <h5 class="card-title mb-4">Полезные разделы</h5>
                    <div class="row">
                        <div class="col-md-3 mb-3">
                            <div class="list-group">
                                <a href="/how-to-buy.php" class="list-group-item list-group-item-action">
                                    <i class="fas fa-shopping-cart me-2"></i>Как купить
                                </a>
                                <a href="/how-to-download.php" class="list-group-item list-group-item-action">
                                    <i class="fas fa-download me-2"></i>Как скачать
                                </a>
                                <a href="/refund.php" class="list-group-item list-group-item-action">
                                    <i class="fas fa-undo me-2"></i>Возврат средств
                                </a>
                            </div>
                        </div>
                        <div class="col-md-3 mb-3">
                            <div class="list-group">
                                <a href="/payment-methods.php" class="list-group-item list-group-item-action">
                                    <i class="fas fa-credit-card me-2"></i>Способы оплаты
                                </a>
                                <a href="/contacts.php" class="list-group-item list-group-item-action">
                                    <i class="fas fa-headset me-2"></i>Контакты
                                </a>
                                <a href="/terms.php" class="list-group-item list-group-item-action">
                                    <i class="fas fa-file-contract me-2"></i>Пользовательское соглашение
                                </a>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="alert alert-info">
                                <h6><i class="fas fa-lightbulb me-2"></i>Быстрые ответы</h6>
                                <p class="small mb-2"><strong>Вопрос:</strong> Сколько времени занимает доставка?</p>
                                <p class="small mb-3"><strong>Ответ:</strong> Все цифровые товары доставляются мгновенно после оплаты.</p>
                                
                                <p class="small mb-2"><strong>Вопрос:</strong> Можно ли вернуть товар?</p>
                                <p class="small mb-0"><strong>Ответ:</strong> Да, в течение 14 дней при наличии технических проблем.</p>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Модальное окно для вопроса -->
<div class="modal fade" id="askQuestionModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Задать вопрос службе поддержки</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form id="askQuestionForm">
                <div class="modal-body">
                    <div class="mb-3">
                        <label for="userName" class="form-label">Ваше имя *</label>
                        <input type="text" class="form-control" id="userName" required>
                    </div>
                    
                    <div class="mb-3">
                        <label for="userEmail" class="form-label">Email для ответа *</label>
                        <input type="email" class="form-control" id="userEmail" required>
                    </div>
                    
                    <div class="mb-3">
                        <label for="questionCategory" class="form-label">Категория вопроса</label>
                        <select class="form-select" id="questionCategory">
                            <option value="">Выберите категорию</option>
                            <option value="buying">Покупка товаров</option>
                            <option value="download">Скачивание и активация</option>
                            <option value="payment">Оплата и счета</option>
                            <option value="technical">Технические вопросы</option>
                            <option value="refund">Возврат и гарантия</option>
                            <option value="account">Аккаунт и профиль</option>
                            <option value="other">Другое</option>
                        </select>
                    </div>
                    
                    <div class="mb-3">
                        <label for="questionText" class="form-label">Ваш вопрос *</label>
                        <textarea class="form-control" id="questionText" rows="4" required
                                  placeholder="Опишите подробно ваш вопрос или проблему..."></textarea>
                        <div class="form-text">Чем подробнее вы опишете проблему, тем быстрее мы сможем помочь</div>
                    </div>
                    
                    <div class="mb-3">
                        <label for="orderNumber" class="form-label">Номер заказа (если есть)</label>
                        <input type="text" class="form-control" id="orderNumber" 
                               placeholder="ORD-2023-XXXXX">
                    </div>
                    
                    <div class="form-check mb-3">
                        <input type="checkbox" class="form-check-input" id="publishQuestion" checked>
                        <label class="form-check-label" for="publishQuestion">
                            Разрешаю опубликовать вопрос и ответ в разделе FAQ (без личной информации)
                        </label>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Отмена</button>
                    <button type="submit" class="btn btn-primary">Отправить вопрос</button>
                </div>
            </form>
        </div>
    </div>
</div>

<style>
.faq-category-card {
    transition: all 0.3s ease;
    border: 1px solid rgba(0,0,0,0.1);
}

.faq-category-card:hover {
    transform: translateY(-5px);
    box-shadow: 0 10px 20px rgba(0,0,0,0.1);
    border-color: var(--primary-color);
}

.category-icon {
    color: var(--primary-color);
}

.question-number {
    display: inline-block;
    width: 30px;
    height: 30px;
    background-color: var(--primary-color);
    color: white;
    border-radius: 50%;
    text-align: center;
    line-height: 30px;
    font-weight: bold;
}

.accordion-button:not(.collapsed) .question-number {
    background-color: white;
    color: var(--primary-color);
}

.faq-answer {
    line-height: 1.6;
}
</style>

<script>
document.addEventListener('DOMContentLoaded', function() {
    // Поиск по FAQ
    const faqSearch = document.getElementById('faqSearch');
    const searchFaqBtn = document.getElementById('searchFaqBtn');
    
    function searchFaq(query) {
        if (!query.trim()) return;
        
        query = query.toLowerCase();
        let found = false;
        
        // Ищем в вопросах
        document.querySelectorAll('.accordion-button').forEach(button => {
            const question = button.textContent.toLowerCase();
            const parent = button.closest('.accordion-item');
            
            if (question.includes(query)) {
                // Раскрываем аккордеон
                const collapseId = button.getAttribute('data-bs-target').substring(1);
                const collapse = document.getElementById(collapseId);
                if (collapse) {
                    new bootstrap.Collapse(collapse, { show: true });
                }
                
                // Подсвечиваем
                button.scrollIntoView({ behavior: 'smooth', block: 'center' });
                button.style.backgroundColor = 'rgba(13, 110, 253, 0.1)';
                setTimeout(() => button.style.backgroundColor = '', 2000);
                
                found = true;
            }
        });
        
        // Ищем в ответах
        if (!found) {
            document.querySelectorAll('.accordion-body').forEach(body => {
                const answer = body.textContent.toLowerCase();
                const button = body.closest('.accordion-item').querySelector('.accordion-button');
                
                if (answer.includes(query)) {
                    // Раскрываем аккордеон
                    const collapse = body.closest('.accordion-collapse');
                    if (collapse) {
                        new bootstrap.Collapse(collapse, { show: true });
                    }
                    
                    // Подсвечиваем текст
                    highlightText(body, query);
                    button.scrollIntoView({ behavior: 'smooth', block: 'center' });
                    
                    found = true;
                }
            });
        }
        
        if (!found) {
            showNotification('Ничего не найдено по вашему запросу', 'info');
        }
    }
    
    function highlightText(element, query) {
        const text = element.innerHTML;
        const regex = new RegExp(`(${query})`, 'gi');
        const highlighted = text.replace(regex, '<mark>$1</mark>');
        element.innerHTML = highlighted;
    }
    
    if (searchFaqBtn && faqSearch) {
        searchFaqBtn.addEventListener('click', () => searchFaq(faqSearch.value));
        
        faqSearch.addEventListener('keypress', (e) => {
            if (e.key === 'Enter') {
                searchFaq(faqSearch.value);
            }
        });
    }
    
    // Отправка вопроса
    const askQuestionForm = document.getElementById('askQuestionForm');
    if (askQuestionForm) {
        askQuestionForm.addEventListener('submit', function(e) {
            e.preventDefault();
            
            const formData = {
                name: document.getElementById('userName').value,
                email: document.getElementById('userEmail').value,
                category: document.getElementById('questionCategory').value,
                question: document.getElementById('questionText').value,
                order_number: document.getElementById('orderNumber').value,
                publish: document.getElementById('publishQuestion').checked
            };
            
            fetch('/api/ask-question.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-Token': window.App.csrfToken
                },
                body: JSON.stringify(formData)
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    showNotification('Ваш вопрос отправлен! Мы ответим в течение 24 часов.', 'success');
                    const modal = bootstrap.Modal.getInstance(document.getElementById('askQuestionModal'));
                    modal.hide();
                    askQuestionForm.reset();
                } else {
                    showNotification(data.error || 'Ошибка отправки вопроса', 'error');
                }
            });
        });
    }
    
    // Отслеживание просмотров вопросов
    document.querySelectorAll('.accordion-button').forEach(button => {
        button.addEventListener('click', function() {
            const questionId = this.getAttribute('data-bs-target').match(/\d+$/);
            if (questionId) {
                // Отправляем статистику просмотра
                fetch('/api/faq-view.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json'
                    },
                    body: JSON.stringify({ question_id: questionId[0] })
                });
            }
        });
    });
});
</script>

<?php include 'templates/footer.php'; ?>