<?php
class Security {
    
    // Защита от CSRF
    public static function generateCSRFToken() {
        if (empty($_SESSION['csrf_token'])) {
            $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
        }
        return $_SESSION['csrf_token'];
    }
    
    public static function validateCSRFToken($token) {
        if (!isset($_SESSION['csrf_token']) || !hash_equals($_SESSION['csrf_token'], $token)) {
            return false;
        }
        return true;
    }
    
    // Защита от брутфорса
    public static function checkLoginAttempts($ip) {
        $db = db();
        
        $stmt = $db->prepare("
            SELECT * FROM login_attempts 
            WHERE ip_address = ? AND is_blocked = 1 AND blocked_until > NOW()
        ");
        $stmt->bind_param("s", $ip);
        $stmt->execute();
        
        if ($stmt->get_result()->num_rows > 0) {
            return false; // IP заблокирован
        }
        
        return true;
    }
    
    public static function recordLoginAttempt($ip, $success) {
        $db = db();
        
        if ($success) {
            // Успешный вход - очищаем попытки
            $stmt = $db->prepare("DELETE FROM login_attempts WHERE ip_address = ?");
            $stmt->bind_param("s", $ip);
            $stmt->execute();
        } else {
            // Неудачная попытка
            $stmt = $db->prepare("
                INSERT INTO login_attempts (ip_address, attempts, last_attempt) 
                VALUES (?, 1, NOW()) 
                ON DUPLICATE KEY UPDATE 
                attempts = attempts + 1, 
                last_attempt = NOW(),
                is_blocked = IF(attempts >= 5, 1, 0),
                blocked_until = IF(attempts >= 5, DATE_ADD(NOW(), INTERVAL 30 MINUTE), NULL)
            ");
            $stmt->bind_param("s", $ip);
            $stmt->execute();
        }
    }
    
    // Валидация файлов
    public static function validateUploadedFile($file, $allowedTypes = [], $maxSize = MAX_FILE_SIZE) {
        $errors = [];
        
        // Проверка размера
        if ($file['size'] > $maxSize) {
            $errors[] = "Файл слишком большой. Максимум: " . ($maxSize / 1024 / 1024) . "MB";
        }
        
        // Проверка типа файла
        $finfo = finfo_open(FILEINFO_MIME_TYPE);
        $mime = finfo_file($finfo, $file['tmp_name']);
        finfo_close($finfo);
        
        if (!empty($allowedTypes) && !in_array($mime, $allowedTypes)) {
            $errors[] = "Недопустимый тип файла";
        }
        
        // Проверка на вредоносный код (базовая)
        if (self::containsMaliciousContent($file['tmp_name'])) {
            $errors[] = "Файл содержит потенциально опасный контент";
        }
        
        return $errors;
    }
    
    private static function containsMaliciousContent($filePath) {
        $dangerousPatterns = [
            '/<\?php/i',
            '/eval\s*\(/i',
            '/base64_decode/i',
            '/system\s*\(/i',
            '/shell_exec/i',
            '/passthru/i',
            '/exec\s*\(/i',
            '/proc_open/i'
        ];
        
        $content = file_get_contents($filePath);
        
        foreach ($dangerousPatterns as $pattern) {
            if (preg_match($pattern, $content)) {
                return true;
            }
        }
        
        return false;
    }
    
    // Защита от SQL инъекций (дополнительно к prepared statements)
    public static function sanitizeSQL($input) {
        $dangerous = [
            "'", "\"", ";", "--", "/*", "*/", 
            "UNION", "SELECT", "INSERT", "DELETE", 
            "UPDATE", "DROP", "CREATE", "ALTER"
        ];
        
        return str_ireplace($dangerous, '', $input);
    }
    
    // XSS защита
    public static function xssClean($data) {
        if (is_array($data)) {
            return array_map([self::class, 'xssClean'], $data);
        }
        
        $data = htmlspecialchars($data, ENT_QUOTES | ENT_HTML5, 'UTF-8');
        
        // Удаляем опасные атрибуты
        $data = preg_replace('/javascript:/i', '', $data);
        $data = preg_replace('/on\w+=/i', 'data-', $data);
        
        return $data;
    }
    
    // Генерация безопасного имени файла
    public static function safeFileName($filename) {
        $filename = preg_replace('/[^\w\.\-]/', '_', $filename);
        $filename = preg_replace('/_+/', '_', $filename);
        $filename = trim($filename, '_');
        
        // Добавляем временную метку для уникальности
        $ext = pathinfo($filename, PATHINFO_EXTENSION);
        $name = pathinfo($filename, PATHINFO_FILENAME);
        
        return substr($name, 0, 50) . '_' . time() . '.' . $ext;
    }
    
    // Проверка HTTPS
    public static function forceHTTPS() {
        if (!isset($_SERVER['HTTPS']) || $_SERVER['HTTPS'] !== 'on') {
            if (!headers_sent()) {
                header("Location: https://" . $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI']);
                exit;
            }
        }
    }
    
    // Заголовки безопасности
    public static function securityHeaders() {
        header("X-Frame-Options: DENY");
        header("X-XSS-Protection: 1; mode=block");
        header("X-Content-Type-Options: nosniff");
        header("Referrer-Policy: strict-origin-when-cross-origin");
        header("Content-Security-Policy: default-src 'self'; script-src 'self' 'unsafe-inline'; style-src 'self' 'unsafe-inline';");
        
        if (!headers_sent()) {
            header_remove("X-Powered-By");
            header_remove("Server");
        }
    }
}
?>