<?php
require_once 'includes/config.php';
require_once 'includes/functions.php';

$title = "Новинки";
$theme = isset($_COOKIE['theme']) ? $_COOKIE['theme'] : 'light';

$db = db();

// Получаем новые товары (последние добавленные)
$page = isset($_GET['page']) ? max(1, (int)$_GET['page']) : 1;
$limit = 12;
$offset = ($page - 1) * $limit;

// Дата для "новинок" (товары добавленные за последние 30 дней)
$new_date = date('Y-m-d H:i:s', strtotime('-30 days'));

$products = $db->query("
    SELECT SQL_CALC_FOUND_ROWS 
        p.*, 
        c.name as category_name,
        c.slug as category_slug,
        COALESCE(p.discount_price, p.price) as final_price,
        DATEDIFF(NOW(), p.created_at) as days_ago
    FROM products p
    LEFT JOIN categories c ON p.category_id = c.id
    WHERE p.status = 'active' 
    AND p.created_at >= '$new_date'
    ORDER BY p.created_at DESC
    LIMIT $limit OFFSET $offset
");

// Общее количество
$total_result = $db->query("SELECT FOUND_ROWS() as total");
$total_row = $total_result->fetch_assoc();
$total_products = $total_row['total'];
$total_pages = ceil($total_products / $limit);

// Получаем последние 5 новинок для баннера
$latest_products = $db->query("
    SELECT p.*, c.name as category_name
    FROM products p
    LEFT JOIN categories c ON p.category_id = c.id
    WHERE p.status = 'active'
    ORDER BY p.created_at DESC
    LIMIT 5
");

include 'templates/header.php';
?>

<div class="container py-5">
    <!-- Заголовок с описанием -->
    <div class="row mb-5">
        <div class="col-12">
            <h1 class="display-5 mb-3">
                <i class="fas fa-bolt text-danger me-2"></i>Новинки
            </h1>
            <p class="lead text-muted">
                Самые свежие поступления в нашем магазине. 
                Будьте в курсе новых цифровых товаров, которые только что появились в продаже.
            </p>
        </div>
    </div>
    
    <!-- Баннер с последними поступлениями -->
    <div class="row mb-5">
        <div class="col-12">
            <div class="card bg-gradient-danger text-white">
                <div class="card-body">
                    <div class="row align-items-center">
                        <div class="col-md-8">
                            <h3 class="card-title mb-3">
                                <i class="fas fa-rocket me-2"></i>Только что добавлены!
                            </h3>
                            <p class="card-text mb-0">
                                Самые свежие товары, которые появились в продаже буквально вчера. 
                                Успейте приобрести первыми!
                            </p>
                        </div>
                        <div class="col-md-4 text-md-end">
                            <div class="badge bg-white text-danger fs-5 p-3">
                                <i class="fas fa-clock me-2"></i>Новые поступления
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Последние 5 товаров -->
    <?php if ($latest_products->num_rows > 0): ?>
    <div class="row mb-5">
        <div class="col-12">
            <h4 class="mb-4">
                <i class="fas fa-history me-2"></i>Последние поступления
            </h4>
            
            <div class="row">
                <?php 
                $latest_counter = 1;
                while ($product = $latest_products->fetch_assoc()): 
                    $days_ago = floor((time() - strtotime($product['created_at'])) / (60 * 60 * 24));
                    $time_label = $days_ago == 0 ? 'Сегодня' : ($days_ago == 1 ? 'Вчера' : $days_ago . ' дней назад');
                ?>
                <div class="col-md-6 col-lg-4 col-xl-3 mb-4">
                    <div class="card h-100 new-product-card border-danger border-2">
                        <!-- Бейдж "Новинка" -->
                        <div class="position-absolute top-0 start-0 m-2">
                            <span class="badge bg-danger">
                                <i class="fas fa-star me-1"></i>НОВИНКА
                            </span>
                        </div>
                        
                        <!-- Бейдж времени -->
                        <div class="position-absolute top-0 end-0 m-2">
                            <span class="badge bg-info">
                                <i class="fas fa-clock me-1"></i><?php echo $time_label; ?>
                            </span>
                        </div>
                        
                        <!-- Изображение товара -->
                        <div class="card-img-container" style="height: 180px;">
                            <img src="<?php echo $product['image'] ?: '/assets/images/default-product.jpg'; ?>" 
                                 class="card-img-top h-100" 
                                 alt="<?php echo htmlspecialchars($product['name']); ?>"
                                 style="object-fit: cover;">
                        </div>
                        
                        <div class="card-body">
                            <h6 class="card-title">
                                <a href="/product.php?id=<?php echo $product['id']; ?>" class="text-decoration-none">
                                    <?php echo htmlspecialchars(mb_substr($product['name'], 0, 40)) . (mb_strlen($product['name']) > 40 ? '...' : ''); ?>
                                </a>
                            </h6>
                            
                            <p class="card-text text-muted extra-small mb-2">
                                <i class="fas fa-tag me-1"></i>
                                <?php echo htmlspecialchars($product['category_name']); ?>
                            </p>
                            
                            <div class="d-flex justify-content-between align-items-center">
                                <div class="price">
                                    <?php if ($product['discount_price']): ?>
                                        <div class="text-muted small text-decoration-line-through">
                                            <?php echo number_format($product['price'], 0); ?> ₽
                                        </div>
                                        <div class="h6 text-success mb-0">
                                            <?php echo number_format($product['discount_price'], 0); ?> ₽
                                        </div>
                                    <?php else: ?>
                                        <div class="h6 text-primary mb-0">
                                            <?php echo number_format($product['price'], 0); ?> ₽
                                        </div>
                                    <?php endif; ?>
                                </div>
                                
                                <button class="btn btn-sm btn-outline-primary add-to-cart" data-product-id="<?php echo $product['id']; ?>">
                                    <i class="fas fa-cart-plus"></i>
                                </button>
                            </div>
                        </div>
                        
                        <div class="card-footer bg-transparent border-top-0">
                            <div class="d-flex justify-content-between extra-small text-muted">
                                <span>
                                    <i class="fas fa-calendar me-1"></i>
                                    <?php echo date('d.m.Y', strtotime($product['created_at'])); ?>
                                </span>
                                <span>
                                    <i class="fas fa-download me-1"></i>
                                    <?php echo number_format($product['downloads_count']); ?>
                                </span>
                            </div>
                        </div>
                    </div>
                </div>
                <?php 
                $latest_counter++;
                endwhile; 
                ?>
            </div>
        </div>
    </div>
    <?php endif; ?>
    
    <!-- Все новинки -->
    <div class="row mb-4">
        <div class="col-12">
            <h3 class="border-bottom pb-3">
                <i class="fas fa-boxes text-danger me-2"></i>Все новинки
                <span class="badge bg-danger ms-2"><?php echo $total_products; ?> товаров</span>
            </h3>
            <p class="text-muted">
                Товары, добавленные за последние 30 дней. Мы постоянно пополняем ассортимент свежим контентом.
            </p>
        </div>
    </div>
    
    <?php if ($products->num_rows > 0): ?>
        <div class="row row-cols-1 row-cols-md-2 row-cols-lg-3 g-4 mb-5">
            <?php while ($product = $products->fetch_assoc()): 
                $days_ago = $product['days_ago'];
                $is_very_new = $days_ago <= 7; // Очень новые (до 7 дней)
                $time_label = $days_ago == 0 ? 'Сегодня' : ($days_ago == 1 ? 'Вчера' : $days_ago . ' дн. назад');
            ?>
            <div class="col">
                <div class="card h-100 product-card position-relative">
                    <?php if ($is_very_new): ?>
                    <div class="position-absolute top-0 start-0 m-2">
                        <span class="badge bg-danger">
                            <i class="fas fa-fire me-1"></i>ГОРЯЧЕЕ
                        </span>
                    </div>
                    <?php endif; ?>
                    
                    <div class="position-absolute top-0 end-0 m-2">
                        <span class="badge bg-info">
                            <i class="fas fa-clock me-1"></i><?php echo $time_label; ?>
                        </span>
                    </div>
                    
                    <div class="card-img-container">
                        <img src="<?php echo $product['image'] ?: '/assets/images/default-product.jpg'; ?>" 
                             class="card-img-top" 
                             alt="<?php echo htmlspecialchars($product['name']); ?>"
                             loading="lazy">
                    </div>
                    
                    <div class="card-body">
                        <h5 class="card-title">
                            <a href="/product.php?id=<?php echo $product['id']; ?>" class="text-decoration-none">
                                <?php echo htmlspecialchars($product['name']); ?>
                            </a>
                        </h5>
                        
                        <p class="card-text text-muted small">
                            <i class="fas fa-tag me-1"></i>
                            <a href="/catalog.php?category=<?php echo $product['category_id']; ?>" class="text-decoration-none text-muted">
                                <?php echo htmlspecialchars($product['category_name']); ?>
                            </a>
                        </p>
                        
                        <p class="card-text small">
                            <?php echo htmlspecialchars(mb_substr($product['description'], 0, 100)) . (mb_strlen($product['description']) > 100 ? '...' : ''); ?>
                        </p>
                        
                        <div class="d-flex justify-content-between align-items-center mt-3">
                            <div class="price">
                                <?php if ($product['discount_price']): ?>
                                    <div class="text-muted small text-decoration-line-through">
                                        <?php echo number_format($product['price'], 2); ?> ₽
                                    </div>
                                    <div class="h5 text-success mb-0">
                                        <?php echo number_format($product['discount_price'], 2); ?> ₽
                                    </div>
                                <?php else: ?>
                                    <div class="h5 text-primary mb-0">
                                        <?php echo number_format($product['price'], 2); ?> ₽
                                    </div>
                                <?php endif; ?>
                            </div>
                            
                            <div class="btn-group">
                                <a href="/product.php?id=<?php echo $product['id']; ?>" class="btn btn-sm btn-outline-primary">
                                    Подробнее
                                </a>
                                <button class="btn btn-sm btn-primary add-to-cart" data-product-id="<?php echo $product['id']; ?>">
                                    <i class="fas fa-cart-plus"></i>
                                </button>
                            </div>
                        </div>
                    </div>
                    
                    <div class="card-footer bg-transparent">
                        <div class="d-flex justify-content-between small text-muted">
                            <span>
                                <i class="fas fa-calendar me-1"></i>
                                <?php echo date('d.m.Y', strtotime($product['created_at'])); ?>
                            </span>
                            <span>
                                <i class="fas fa-download me-1"></i>
                                <?php echo number_format($product['downloads_count']); ?>
                            </span>
                        </div>
                    </div>
                </div>
            </div>
            <?php endwhile; ?>
        </div>
        
        <!-- Пагинация -->
        <?php if ($total_pages > 1): ?>
        <nav aria-label="Навигация по страницам">
            <ul class="pagination justify-content-center">
                <?php if ($page > 1): ?>
                <li class="page-item">
                    <a class="page-link" href="?page=<?php echo $page - 1; ?>">
                        <i class="fas fa-chevron-left"></i>
                    </a>
                </li>
                <?php endif; ?>
                
                <?php
                $start = max(1, $page - 2);
                $end = min($total_pages, $page + 2);
                
                if ($start > 1) {
                    echo '<li class="page-item"><a class="page-link" href="?page=1">1</a></li>';
                    if ($start > 2) {
                        echo '<li class="page-item disabled"><span class="page-link">...</span></li>';
                    }
                }
                
                for ($i = $start; $i <= $end; $i++):
                ?>
                <li class="page-item <?php echo $i == $page ? 'active' : ''; ?>">
                    <a class="page-link" href="?page=<?php echo $i; ?>">
                        <?php echo $i; ?>
                    </a>
                </li>
                <?php endfor; ?>
                
                if ($end < $total_pages) {
                    if ($end < $total_pages - 1) {
                        echo '<li class="page-item disabled"><span class="page-link">...</span></li>';
                    }
                    echo '<li class="page-item"><a class="page-link" href="?page=' . $total_pages . '">' . $total_pages . '</a></li>';
                }
                ?>
                
                <?php if ($page < $total_pages): ?>
                <li class="page-item">
                    <a class="page-link" href="?page=<?php echo $page + 1; ?>">
                        <i class="fas fa-chevron-right"></i>
                    </a>
                </li>
                <?php endif; ?>
            </ul>
        </nav>
        <?php endif; ?>
        
    <?php else: ?>
        <div class="card">
            <div class="card-body text-center py-5">
                <i class="fas fa-box-open fa-4x text-muted mb-4"></i>
                <h3 class="text-muted mb-3">Новинки не найдены</h3>
                <p class="text-muted mb-4">
                    В настоящее время новых товаров нет. Загляните позже или посмотрите другие разделы.
                </p>
                <div class="d-grid gap-2 d-md-block">
                    <a href="/catalog.php" class="btn btn-primary me-md-2">
                        <i class="fas fa-shopping-bag me-2"></i>Каталог товаров
                    </a>
                    <a href="/popular.php" class="btn btn-outline-primary">
                        <i class="fas fa-fire me-2"></i>Популярные товары
                    </a>
                </div>
            </div>
        </div>
    <?php endif; ?>
    
    <!-- Уведомление о новинках -->
    <div class="row mt-5">
        <div class="col-12">
            <div class="card bg-light border-danger">
                <div class="card-body">
                    <div class="row align-items-center">
                        <div class="col-md-8">
                            <h5 class="card-title text-danger mb-3">
                                <i class="fas fa-bell me-2"></i>Узнавайте о новинках первыми!
                            </h5>
                            <p class="card-text mb-0">
                                Подпишитесь на уведомления о новых поступлениях. Мы будем сообщать вам 
                                о свежих товарах в выбранных категориях.
                            </p>
                        </div>
                        <div class="col-md-4 text-md-end">
                            <button class="btn btn-danger" data-bs-toggle="modal" data-bs-target="#subscribeNewModal">
                                <i class="fas fa-envelope me-2"></i>Подписаться
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Модальное окно подписки на новинки -->
<div class="modal fade" id="subscribeNewModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Подписка на новинки</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form id="subscribeNewForm">
                <div class="modal-body">
                    <div class="mb-3">
                        <label for="subscribeEmail" class="form-label">Ваш email *</label>
                        <input type="email" class="form-control" id="subscribeEmail" required>
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label">Категории для уведомлений</label>
                        <?php
                        $cats = $db->query("SELECT id, name FROM categories WHERE is_active = 1 ORDER BY name");
                        while ($cat = $cats->fetch_assoc()):
                        ?>
                        <div class="form-check">
                            <input class="form-check-input" type="checkbox" 
                                   id="cat_<?php echo $cat['id']; ?>" 
                                   name="categories[]" 
                                   value="<?php echo $cat['id']; ?>" 
                                   checked>
                            <label class="form-check-label" for="cat_<?php echo $cat['id']; ?>">
                                <?php echo htmlspecialchars($cat['name']); ?>
                            </label>
                        </div>
                        <?php endwhile; ?>
                    </div>
                    
                    <div class="form-check mb-3">
                        <input type="checkbox" class="form-check-input" id="frequencyWeekly" name="frequency" value="weekly" checked>
                        <label class="form-check-label" for="frequencyWeekly">
                            Еженедельная рассылка
                        </label>
                    </div>
                    
                    <div class="form-check">
                        <input type="checkbox" class="form-check-input" id="frequencyInstant" name="frequency" value="instant">
                        <label class="form-check-label" for="frequencyInstant">
                            Мгновенные уведомления о новых товарах
                        </label>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Отмена</button>
                    <button type="submit" class="btn btn-primary">Подписаться</button>
                </div>
            </form>
        </div>
    </div>
</div>

<style>
.bg-gradient-danger {
    background: linear-gradient(135deg, #f093fb 0%, #f5576c 100%) !important;
}

.new-product-card {
    transition: all 0.3s ease;
    border-width: 2px !important;
}

.new-product-card:hover {
    transform: translateY(-5px);
    box-shadow: 0 10px 20px rgba(220, 53, 69, 0.2);
}

.extra-small {
    font-size: 0.8rem;
}
</style>

<script>
document.addEventListener('DOMContentLoaded', function() {
    // Подписка на новинки
    const subscribeForm = document.getElementById('subscribeNewForm');
    if (subscribeForm) {
        subscribeForm.addEventListener('submit', function(e) {
            e.preventDefault();
            
            const email = document.getElementById('subscribeEmail').value;
            const categories = Array.from(document.querySelectorAll('input[name="categories[]"]:checked'))
                .map(cb => cb.value);
            const frequency = document.querySelector('input[name="frequency"]:checked')?.value || 'weekly';
            
            if (!email) {
                showNotification('Введите email', 'error');
                return;
            }
            
            fetch('/api/subscribe-new.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-Token': window.App.csrfToken
                },
                body: JSON.stringify({
                    email: email,
                    categories: categories,
                    frequency: frequency
                })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    showNotification('Вы успешно подписались на новинки!', 'success');
                    const modal = bootstrap.Modal.getInstance(document.getElementById('subscribeNewModal'));
                    modal.hide();
                    subscribeForm.reset();
                } else {
                    showNotification(data.error || 'Ошибка подписки', 'error');
                }
            });
        });
    }
    
    // Анимация карточек новинок
    document.querySelectorAll('.new-product-card').forEach(card => {
        card.addEventListener('mouseenter', function() {
            this.style.boxShadow = '0 15px 30px rgba(220, 53, 69, 0.3)';
        });
        
        card.addEventListener('mouseleave', function() {
            this.style.boxShadow = '';
        });
    });
});
</script>

<?php include 'templates/footer.php'; ?>