<?php
require_once 'includes/config.php';
require_once 'includes/functions.php';

$title = "Популярные товары";
$theme = isset($_COOKIE['theme']) ? $_COOKIE['theme'] : 'light';

$db = db();

// Получаем популярные товары (по количеству скачиваний)
$page = isset($_GET['page']) ? max(1, (int)$_GET['page']) : 1;
$limit = 12;
$offset = ($page - 1) * $limit;

$products = $db->query("
    SELECT SQL_CALC_FOUND_ROWS 
        p.*, 
        c.name as category_name,
        c.slug as category_slug,
        COALESCE(p.discount_price, p.price) as final_price
    FROM products p
    LEFT JOIN categories c ON p.category_id = c.id
    WHERE p.status = 'active'
    ORDER BY p.downloads_count DESC, p.created_at DESC
    LIMIT $limit OFFSET $offset
");

// Общее количество
$total_result = $db->query("SELECT FOUND_ROWS() as total");
$total_row = $total_result->fetch_assoc();
$total_products = $total_row['total'];
$total_pages = ceil($total_products / $limit);

// Получаем топ-5 самых популярных товаров для баннера
$top_products = $db->query("
    SELECT p.*, c.name as category_name
    FROM products p
    LEFT JOIN categories c ON p.category_id = c.id
    WHERE p.status = 'active'
    ORDER BY p.downloads_count DESC
    LIMIT 5
");

include 'templates/header.php';
?>

<div class="container py-5">
    <!-- Заголовок с описанием -->
    <div class="row mb-5">
        <div class="col-12">
            <h1 class="display-5 mb-3">
                <i class="fas fa-fire text-warning me-2"></i>Популярные товары
            </h1>
            <p class="lead text-muted">
                Товары, которые выбирают большинство наших покупателей. 
                Проверенное качество и востребованный контент.
            </p>
        </div>
    </div>
    
    <!-- Баннер с топ-5 товаров -->
    <div class="row mb-5">
        <div class="col-12">
            <div class="card bg-gradient-primary text-white">
                <div class="card-body">
                    <div class="row align-items-center">
                        <div class="col-md-8">
                            <h3 class="card-title mb-3">
                                <i class="fas fa-crown me-2"></i>Топ-5 самых популярных
                            </h3>
                            <p class="card-text mb-0">
                                Самые скачиваемые товары за все время. Эти продукты уже оценили тысячи покупателей.
                            </p>
                        </div>
                        <div class="col-md-4 text-md-end">
                            <div class="badge bg-white text-primary fs-5 p-3">
                                <i class="fas fa-trophy me-2"></i>Лидеры продаж
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Топ-5 товаров -->
    <?php if ($top_products->num_rows > 0): ?>
    <div class="row mb-5">
        <?php 
        $top_counter = 1;
        while ($product = $top_products->fetch_assoc()): 
            $badge_class = $top_counter == 1 ? 'bg-warning' : ($top_counter == 2 ? 'bg-secondary' : ($top_counter == 3 ? 'bg-danger' : 'bg-dark'));
        ?>
        <div class="col-lg-6 mb-4">
            <div class="card h-100 shadow-sm">
                <div class="card-body">
                    <div class="d-flex">
                        <!-- Номер в топе -->
                        <div class="flex-shrink-0 me-3">
                            <div class="top-badge <?php echo $badge_class; ?> text-white rounded-circle d-flex align-items-center justify-content-center" 
                                 style="width: 60px; height: 60px; font-size: 24px; font-weight: bold;">
                                <?php echo $top_counter; ?>
                            </div>
                        </div>
                        
                        <!-- Информация о товаре -->
                        <div class="flex-grow-1">
                            <div class="d-flex justify-content-between align-items-start mb-2">
                                <h5 class="card-title mb-0">
                                    <a href="/product.php?id=<?php echo $product['id']; ?>" class="text-decoration-none">
                                        <?php echo htmlspecialchars($product['name']); ?>
                                    </a>
                                </h5>
                                <span class="badge bg-primary">
                                    <i class="fas fa-download me-1"></i>
                                    <?php echo number_format($product['downloads_count']); ?>
                                </span>
                            </div>
                            
                            <p class="card-text text-muted small mb-2">
                                <i class="fas fa-tag me-1"></i>
                                <?php echo htmlspecialchars($product['category_name']); ?>
                            </p>
                            
                            <p class="card-text small">
                                <?php echo htmlspecialchars(mb_substr($product['description'], 0, 100)) . (mb_strlen($product['description']) > 100 ? '...' : ''); ?>
                            </p>
                            
                            <div class="d-flex justify-content-between align-items-center">
                                <div class="price">
                                    <?php if ($product['discount_price']): ?>
                                        <span class="text-muted text-decoration-line-through me-2">
                                            <?php echo number_format($product['price'], 2); ?> ₽
                                        </span>
                                        <span class="h5 text-success mb-0">
                                            <?php echo number_format($product['discount_price'], 2); ?> ₽
                                        </span>
                                    <?php else: ?>
                                        <span class="h5 text-primary mb-0">
                                            <?php echo number_format($product['price'], 2); ?> ₽
                                        </span>
                                    <?php endif; ?>
                                </div>
                                
                                <div class="btn-group">
                                    <a href="/product.php?id=<?php echo $product['id']; ?>" class="btn btn-sm btn-outline-primary">
                                        Подробнее
                                    </a>
                                    <button class="btn btn-sm btn-primary add-to-cart" data-product-id="<?php echo $product['id']; ?>">
                                        <i class="fas fa-cart-plus"></i>
                                    </button>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        <?php 
        $top_counter++;
        endwhile; 
        ?>
    </div>
    <?php endif; ?>
    
    <!-- Все популярные товары -->
    <div class="row mb-4">
        <div class="col-12">
            <h3 class="border-bottom pb-3">
                <i class="fas fa-star text-warning me-2"></i>Все популярные товары
                <span class="badge bg-primary ms-2"><?php echo $total_products; ?> товаров</span>
            </h3>
        </div>
    </div>
    
    <?php if ($products->num_rows > 0): ?>
        <div class="row row-cols-1 row-cols-md-2 row-cols-lg-3 g-4 mb-5">
            <?php while ($product = $products->fetch_assoc()): ?>
            <div class="col">
                <?php include 'templates/product-card.php'; ?>
            </div>
            <?php endwhile; ?>
        </div>
        
        <!-- Пагинация -->
        <?php if ($total_pages > 1): ?>
        <nav aria-label="Навигация по страницам">
            <ul class="pagination justify-content-center">
                <?php if ($page > 1): ?>
                <li class="page-item">
                    <a class="page-link" href="?page=<?php echo $page - 1; ?>">
                        <i class="fas fa-chevron-left"></i>
                    </a>
                </li>
                <?php endif; ?>
                
                <?php
                $start = max(1, $page - 2);
                $end = min($total_pages, $page + 2);
                
                if ($start > 1) {
                    echo '<li class="page-item"><a class="page-link" href="?page=1">1</a></li>';
                    if ($start > 2) {
                        echo '<li class="page-item disabled"><span class="page-link">...</span></li>';
                    }
                }
                
                for ($i = $start; $i <= $end; $i++):
                ?>
                <li class="page-item <?php echo $i == $page ? 'active' : ''; ?>">
                    <a class="page-link" href="?page=<?php echo $i; ?>">
                        <?php echo $i; ?>
                    </a>
                </li>
                <?php endfor; ?>
                
                if ($end < $total_pages) {
                    if ($end < $total_pages - 1) {
                        echo '<li class="page-item disabled"><span class="page-link">...</span></li>';
                    }
                    echo '<li class="page-item"><a class="page-link" href="?page=' . $total_pages . '">' . $total_pages . '</a></li>';
                }
                ?>
                
                <?php if ($page < $total_pages): ?>
                <li class="page-item">
                    <a class="page-link" href="?page=<?php echo $page + 1; ?>">
                        <i class="fas fa-chevron-right"></i>
                    </a>
                </li>
                <?php endif; ?>
            </ul>
        </nav>
        <?php endif; ?>
        
    <?php else: ?>
        <div class="card">
            <div class="card-body text-center py-5">
                <i class="fas fa-fire fa-4x text-muted mb-4"></i>
                <h3 class="text-muted mb-3">Популярные товары не найдены</h3>
                <p class="text-muted mb-4">
                    В настоящее время статистика популярности формируется
                </p>
                <a href="/catalog.php" class="btn btn-primary">
                    <i class="fas fa-shopping-bag me-2"></i>Перейти в каталог
                </a>
            </div>
        </div>
    <?php endif; ?>
    
    <!-- Почему выбирают популярные товары -->
    <div class="row mt-5">
        <div class="col-12">
            <div class="card bg-light">
                <div class="card-body">
                    <h5 class="card-title mb-4">
                        <i class="fas fa-question-circle me-2"></i>Почему стоит выбирать популярные товары?
                    </h5>
                    
                    <div class="row">
                        <div class="col-md-4 mb-3">
                            <div class="d-flex">
                                <div class="flex-shrink-0">
                                    <i class="fas fa-check-circle text-success fa-2x me-3"></i>
                                </div>
                                <div class="flex-grow-1">
                                    <h6>Проверенное качество</h6>
                                    <p class="small text-muted mb-0">
                                        Эти товары уже купили тысячи пользователей и подтвердили их качество
                                    </p>
                                </div>
                            </div>
                        </div>
                        
                        <div class="col-md-4 mb-3">
                            <div class="d-flex">
                                <div class="flex-shrink-0">
                                    <i class="fas fa-comments text-primary fa-2x me-3"></i>
                                </div>
                                <div class="flex-grow-1">
                                    <h6>Подробные отзывы</h6>
                                    <p class="small text-muted mb-0">
                                        На популярные товары больше всего отзывов и оценок от реальных покупателей
                                    </p>
                                </div>
                            </div>
                        </div>
                        
                        <div class="col-md-4 mb-3">
                            <div class="d-flex">
                                <div class="flex-shrink-0">
                                    <i class="fas fa-headset text-warning fa-2x me-3"></i>
                                </div>
                                <div class="flex-grow-1">
                                    <h6>Техническая поддержка</h6>
                                    <p class="small text-muted mb-0">
                                        Авторы активно поддерживают популярные продукты и оперативно решают проблемы
                                    </p>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<style>
.bg-gradient-primary {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%) !important;
}

.top-badge {
    box-shadow: 0 4px 6px rgba(0,0,0,0.1);
}

.card:hover .top-badge {
    transform: scale(1.1);
    transition: transform 0.3s;
}
</style>

<script>
document.addEventListener('DOMContentLoaded', function() {
    // Анимация для топ-товаров
    document.querySelectorAll('.top-badge').forEach((badge, index) => {
        badge.addEventListener('mouseenter', function() {
            this.style.transform = 'scale(1.2) rotate(5deg)';
        });
        
        badge.addEventListener('mouseleave', function() {
            this.style.transform = 'scale(1) rotate(0)';
        });
    });
});
</script>

<?php include 'templates/footer.php'; ?>