<?php
require_once 'includes/config.php';
require_once 'includes/functions.php';
require_once 'includes/auth.php';

// Если пользователь уже авторизован, перенаправляем
if (isLoggedIn()) {
    header('Location: /profile.php');
    exit;
}

$errors = [];
$success = false;

// Обработка формы регистрации
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $email = sanitize($_POST['email'] ?? '');
    $username = sanitize($_POST['username'] ?? '');
    $password = $_POST['password'] ?? '';
    $password_confirm = $_POST['password_confirm'] ?? '';
    $agree = isset($_POST['agree']);
    
    // Валидация
    if (empty($email)) {
        $errors['email'] = 'Email обязателен';
    } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $errors['email'] = 'Некорректный email';
    } elseif (strlen($email) > 100) {
        $errors['email'] = 'Email слишком длинный';
    }
    
    if (empty($username)) {
        $errors['username'] = 'Имя пользователя обязательно';
    } elseif (strlen($username) < 3) {
        $errors['username'] = 'Имя пользователя должно быть не менее 3 символов';
    } elseif (strlen($username) > 50) {
        $errors['username'] = 'Имя пользователя слишком длинное';
    } elseif (!preg_match('/^[a-zA-Z0-9_]+$/', $username)) {
        $errors['username'] = 'Имя пользователя может содержать только буквы, цифры и подчеркивания';
    }
    
    if (empty($password)) {
        $errors['password'] = 'Пароль обязателен';
    } elseif (strlen($password) < 6) {
        $errors['password'] = 'Пароль должен быть не менее 6 символов';
    } elseif ($password !== $password_confirm) {
        $errors['password_confirm'] = 'Пароли не совпадают';
    }
    
    if (!$agree) {
        $errors['agree'] = 'Вы должны согласиться с условиями';
    }
    
    // Проверка CSRF токена
    if (!isset($_POST['csrf_token']) || $_POST['csrf_token'] !== $_SESSION['csrf_token']) {
        $errors['form'] = 'Неверный CSRF токен';
    }
    
    // Если ошибок нет, регистрируем пользователя
    if (empty($errors)) {
        $result = registerUser($email, $password, $username);
        
        if (isset($result['success'])) {
            $success = true;
            
            // Отправляем email с подтверждением
            $_SESSION['registration_email'] = $email;
            $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
        } else {
            $errors['form'] = $result['error'];
        }
    }
}

// Генерация CSRF токена
if (empty($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

$title = "Регистрация";
$theme = isset($_COOKIE['theme']) ? $_COOKIE['theme'] : 'light';

include 'templates/header.php';
?>

<div class="container py-5">
    <div class="row justify-content-center">
        <div class="col-md-8 col-lg-6">
            <div class="card shadow">
                <div class="card-header bg-success text-white">
                    <h4 class="mb-0"><i class="fas fa-user-plus me-2"></i>Создание аккаунта</h4>
                </div>
                
                <div class="card-body p-4">
                    <?php if ($success): ?>
                        <div class="alert alert-success">
                            <h5 class="alert-heading"><i class="fas fa-check-circle me-2"></i>Регистрация успешна!</h5>
                            <p>На вашу почту <strong><?php echo htmlspecialchars($_SESSION['registration_email'] ?? ''); ?></strong> было отправлено письмо с ссылкой для активации аккаунта.</p>
                            <p>Пожалуйста, проверьте вашу почту и перейдите по ссылке в письме.</p>
                            <hr>
                            <div class="d-grid gap-2">
                                <a href="/login.php" class="btn btn-outline-success">
                                    <i class="fas fa-sign-in-alt me-2"></i>Перейти к входу
                                </a>
                                <a href="/" class="btn btn-outline-primary">
                                    <i class="fas fa-home me-2"></i>На главную
                                </a>
                            </div>
                        </div>
                    <?php else: ?>
                        <?php if (isset($errors['form'])): ?>
                            <div class="alert alert-danger alert-dismissible fade show">
                                <?php echo $errors['form']; ?>
                                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                            </div>
                        <?php endif; ?>
                        
                        <form method="POST" id="registerForm">
                            <input type="hidden" name="csrf_token" value="<?php echo $_SESSION['csrf_token']; ?>">
                            
                            <div class="row">
                                <div class="col-md-6 mb-3">
                                    <label for="email" class="form-label">
                                        <i class="fas fa-envelope me-1"></i>Email адрес *
                                    </label>
                                    <input type="email" 
                                           class="form-control <?php echo isset($errors['email']) ? 'is-invalid' : ''; ?>" 
                                           id="email" 
                                           name="email" 
                                           value="<?php echo htmlspecialchars($_POST['email'] ?? ''); ?>" 
                                           required
                                           autocomplete="email"
                                           autofocus>
                                    <?php if (isset($errors['email'])): ?>
                                        <div class="invalid-feedback">
                                            <?php echo $errors['email']; ?>
                                        </div>
                                    <?php endif; ?>
                                    <div class="form-text">На этот email будет отправлено письмо для активации</div>
                                </div>
                                
                                <div class="col-md-6 mb-3">
                                    <label for="username" class="form-label">
                                        <i class="fas fa-user me-1"></i>Имя пользователя *
                                    </label>
                                    <input type="text" 
                                           class="form-control <?php echo isset($errors['username']) ? 'is-invalid' : ''; ?>" 
                                           id="username" 
                                           name="username" 
                                           value="<?php echo htmlspecialchars($_POST['username'] ?? ''); ?>" 
                                           required
                                           autocomplete="username">
                                    <?php if (isset($errors['username'])): ?>
                                        <div class="invalid-feedback">
                                            <?php echo $errors['username']; ?>
                                        </div>
                                    <?php endif; ?>
                                    <div class="form-text">От 3 до 50 символов, только буквы, цифры и _</div>
                                </div>
                            </div>
                            
                            <div class="row">
                                <div class="col-md-6 mb-3">
                                    <label for="password" class="form-label">
                                        <i class="fas fa-lock me-1"></i>Пароль *
                                    </label>
                                    <div class="input-group">
                                        <input type="password" 
                                               class="form-control <?php echo isset($errors['password']) ? 'is-invalid' : ''; ?>" 
                                               id="password" 
                                               name="password" 
                                               required
                                               autocomplete="new-password">
                                        <button class="btn btn-outline-secondary" type="button" id="togglePassword">
                                            <i class="fas fa-eye"></i>
                                        </button>
                                    </div>
                                    <?php if (isset($errors['password'])): ?>
                                        <div class="invalid-feedback d-block">
                                            <?php echo $errors['password']; ?>
                                        </div>
                                    <?php endif; ?>
                                    <div class="password-strength mt-2">
                                        <div class="progress" style="height: 5px;">
                                            <div class="progress-bar" id="passwordStrength" role="progressbar" style="width: 0%"></div>
                                        </div>
                                        <small class="text-muted" id="passwordHint">Слабый пароль</small>
                                    </div>
                                </div>
                                
                                <div class="col-md-6 mb-3">
                                    <label for="password_confirm" class="form-label">
                                        <i class="fas fa-lock me-1"></i>Подтверждение пароля *
                                    </label>
                                    <div class="input-group">
                                        <input type="password" 
                                               class="form-control <?php echo isset($errors['password_confirm']) ? 'is-invalid' : ''; ?>" 
                                               id="password_confirm" 
                                               name="password_confirm" 
                                               required
                                               autocomplete="new-password">
                                        <button class="btn btn-outline-secondary" type="button" id="togglePasswordConfirm">
                                            <i class="fas fa-eye"></i>
                                        </button>
                                    </div>
                                    <?php if (isset($errors['password_confirm'])): ?>
                                        <div class="invalid-feedback d-block">
                                            <?php echo $errors['password_confirm']; ?>
                                        </div>
                                    <?php endif; ?>
                                    <div class="form-text">Повторите пароль для подтверждения</div>
                                </div>
                            </div>
                            
                            <div class="mb-4">
                                <div class="form-check">
                                    <input type="checkbox" 
                                           class="form-check-input <?php echo isset($errors['agree']) ? 'is-invalid' : ''; ?>" 
                                           id="agree" 
                                           name="agree"
                                           required>
                                    <label class="form-check-label" for="agree">
                                        Я соглашаюсь с 
                                        <a href="/terms.php" target="_blank" class="text-decoration-none">пользовательским соглашением</a> 
                                        и <a href="/privacy.php" target="_blank" class="text-decoration-none">политикой конфиденциальности</a> *
                                    </label>
                                    <?php if (isset($errors['agree'])): ?>
                                        <div class="invalid-feedback d-block">
                                            <?php echo $errors['agree']; ?>
                                        </div>
                                    <?php endif; ?>
                                </div>
                            </div>
                            
                            <div class="d-grid gap-2">
                                <button type="submit" class="btn btn-success btn-lg">
                                    <i class="fas fa-user-plus me-2"></i>Зарегистрироваться
                                </button>
                                
                                <div class="text-center my-3">
                                    <span class="text-muted">Уже есть аккаунт?</span>
                                </div>
                                
                                <a href="/login.php" class="btn btn-outline-primary">
                                    <i class="fas fa-sign-in-alt me-2"></i>Войти в аккаунт
                                </a>
                            </div>
                        </form>
                    <?php endif; ?>
                </div>
                
                <div class="card-footer text-center bg-light">
                    <p class="mb-0 small text-muted">
                        Регистрируясь, вы получаете доступ к личному кабинету, истории заказов и персональным скидкам
                    </p>
                </div>
            </div>
            
            <!-- Преимущества регистрации -->
            <div class="row mt-4">
                <div class="col-md-4 mb-3">
                    <div class="card h-100">
                        <div class="card-body text-center">
                            <i class="fas fa-bolt fa-2x text-primary mb-3"></i>
                            <h6>Быстрые покупки</h6>
                            <p class="small text-muted">Оформляйте заказы в один клик</p>
                        </div>
                    </div>
                </div>
                <div class="col-md-4 mb-3">
                    <div class="card h-100">
                        <div class="card-body text-center">
                            <i class="fas fa-history fa-2x text-success mb-3"></i>
                            <h6>История заказов</h6>
                            <p class="small text-muted">Все ваши покупки в одном месте</p>
                        </div>
                    </div>
                </div>
                <div class="col-md-4 mb-3">
                    <div class="card h-100">
                        <div class="card-body text-center">
                            <i class="fas fa-percent fa-2x text-warning mb-3"></i>
                            <h6>Скидки</h6>
                            <p class="small text-muted">Персональные скидки для постоянных клиентов</p>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    // Переключение видимости паролей
    const togglePassword = document.getElementById('togglePassword');
    const togglePasswordConfirm = document.getElementById('togglePasswordConfirm');
    const passwordInput = document.getElementById('password');
    const passwordConfirmInput = document.getElementById('password_confirm');
    
    function togglePasswordVisibility(input, button) {
        const type = input.getAttribute('type') === 'password' ? 'text' : 'password';
        input.setAttribute('type', type);
        button.innerHTML = type === 'password' ? '<i class="fas fa-eye"></i>' : '<i class="fas fa-eye-slash"></i>';
    }
    
    if (togglePassword && passwordInput) {
        togglePassword.addEventListener('click', () => togglePasswordVisibility(passwordInput, togglePassword));
    }
    
    if (togglePasswordConfirm && passwordConfirmInput) {
        togglePasswordConfirm.addEventListener('click', () => togglePasswordVisibility(passwordConfirmInput, togglePasswordConfirm));
    }
    
    // Проверка силы пароля
    const passwordStrength = document.getElementById('passwordStrength');
    const passwordHint = document.getElementById('passwordHint');
    
    function checkPasswordStrength(password) {
        let strength = 0;
        
        if (password.length >= 6) strength += 25;
        if (password.length >= 8) strength += 25;
        if (/[A-Z]/.test(password)) strength += 25;
        if (/[0-9]/.test(password)) strength += 25;
        
        return Math.min(strength, 100);
    }
    
    function updatePasswordStrength() {
        const password = passwordInput.value;
        const strength = checkPasswordStrength(password);
        
        passwordStrength.style.width = strength + '%';
        
        if (strength < 30) {
            passwordStrength.className = 'progress-bar bg-danger';
            passwordHint.textContent = 'Слабый пароль';
        } else if (strength < 70) {
            passwordStrength.className = 'progress-bar bg-warning';
            passwordHint.textContent = 'Средний пароль';
        } else {
            passwordStrength.className = 'progress-bar bg-success';
            passwordHint.textContent = 'Сильный пароль';
        }
    }
    
    if (passwordInput) {
        passwordInput.addEventListener('input', updatePasswordStrength);
    }
    
    // Проверка совпадения паролей
    function checkPasswordMatch() {
        const password = passwordInput.value;
        const confirm = passwordConfirmInput.value;
        
        if (confirm.length > 0 && password !== confirm) {
            passwordConfirmInput.classList.add('is-invalid');
        } else {
            passwordConfirmInput.classList.remove('is-invalid');
        }
    }
    
    if (passwordInput && passwordConfirmInput) {
        passwordInput.addEventListener('input', checkPasswordMatch);
        passwordConfirmInput.addEventListener('input', checkPasswordMatch);
    }
    
    // Валидация формы
    const registerForm = document.getElementById('registerForm');
    if (registerForm) {
        registerForm.addEventListener('submit', function(e) {
            const email = document.getElementById('email').value;
            const username = document.getElementById('username').value;
            const password = document.getElementById('password').value;
            const confirm = document.getElementById('password_confirm').value;
            const agree = document.getElementById('agree').checked;
            
            let isValid = true;
            
            // Проверка email
            const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
            if (!emailRegex.test(email)) {
                document.getElementById('email').classList.add('is-invalid');
                isValid = false;
            }
            
            // Проверка имени пользователя
            const usernameRegex = /^[a-zA-Z0-9_]+$/;
            if (username.length < 3 || username.length > 50 || !usernameRegex.test(username)) {
                document.getElementById('username').classList.add('is-invalid');
                isValid = false;
            }
            
            // Проверка пароля
            if (password.length < 6) {
                document.getElementById('password').classList.add('is-invalid');
                isValid = false;
            }
            
            // Проверка совпадения паролей
            if (password !== confirm) {
                document.getElementById('password_confirm').classList.add('is-invalid');
                isValid = false;
            }
            
            // Проверка согласия
            if (!agree) {
                document.getElementById('agree').classList.add('is-invalid');
                isValid = false;
            }
            
            if (!isValid) {
                e.preventDefault();
            }
        });
    }
});
</script>

<?php include 'templates/footer.php'; ?>