<?php

require_once __DIR__ . '/../config/bootstrap.php';
require_once __DIR__ . '/../config/auth.php';
require_once __DIR__ . '/../config/functions.php';

if (!is_admin()) {
    http_response_code(403);
    die("Доступ запрещён");
}

require_once __DIR__ . '/../includes/header.php';

try {

    $stmt = $pdo->query(
        "SELECT o.id, u.email, u.name, o.total, o.status, o.created_at,
                COUNT(oi.id) as items_count
         FROM orders o
         JOIN users u ON o.user_id = u.id
         LEFT JOIN order_items oi ON o.id = oi.order_id
         GROUP BY o.id
         ORDER BY o.created_at DESC"
    );
    $orders = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (Exception $e) {
    die("Ошибка загрузки заказов: " . htmlspecialchars($e->getMessage()));
}
?>

<div class="container my-5">
    <h1 class="mb-4">
        <i class="bi bi-cart-check"></i> Управление заказами
    </h1>

    <div class="glass p-4 card-ui fade-in">
        <div class="table-responsive">
            <table class="table table-hover table-striped align-middle">
                <thead class="table-dark">
                    <tr>
                        <th>#ID</th>
                        <th>Пользователь</th>
                        <th>Email</th>
                        <th>Сумма</th>
                        <th>Товаров</th>
                        <th>Статус</th>
                        <th>Дата создания</th>
                        <th>Действия</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (empty($orders)): ?>
                        <tr>
                            <td colspan="8" class="text-center text-muted py-4">
                                <i class="bi bi-inbox"></i> Нет заказов
                            </td>
                        </tr>
                    <?php else: ?>
                        <?php foreach ($orders as $order): ?>
                        <tr>
                            <td><?= e($order['id']) ?></td>
                            <td><?= e($order['name'] ?? 'Не указано') ?></td>
                            <td><?= e($order['email']) ?></td>
                            <td>
                                <span class="badge bg-success rounded-pill">
                                    <?= number_format($order['total'] ?? 0, 2) ?> ₽
                                </span>
                            </td>
                            <td>
                                <span class="badge bg-primary rounded-pill">
                                    <?= e($order['items_count'] ?? 0) ?> шт.
                                </span>
                            </td>
                            <td>
                                <select class="form-select form-select-sm" 
                                        onchange="updateStatus(<?= $order['id'] ?>, this.value)"
                                        style="width: 150px;">
                                    <?php
                                    $statuses = [
                                        'Новый' => 'bg-secondary',
                                        'В обработке' => 'bg-info',
                                        'Печать' => 'bg-warning',
                                        'Доставляется' => 'bg-primary',
                                        'Доставлен' => 'bg-success',
                                        'Отменен' => 'bg-danger'
                                    ];
                                    
                                    foreach ($statuses as $status => $class):
                                        $selected = ($order['status'] ?? 'Новый') === $status ? 'selected' : '';
                                    ?>
                                        <option value="<?= e($status) ?>" <?= $selected ?>>
                                            <?= e($status) ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </td>
                            <td><?= date('d.m.Y H:i', strtotime($order['created_at'])) ?></td>
                            <td>
                                <div class="btn-group btn-group-sm">
                                    <a href="/admin/order_view.php?id=<?= e($order['id']) ?>" 
                                       class="btn btn-outline-primary">
                                        <i class="bi bi-eye"></i> Просмотр
                                    </a>
                                    <button class="btn btn-outline-danger" 
                                            onclick="deleteOrder(<?= e($order['id']) ?>)">
                                        <i class="bi bi-trash"></i>
                                    </button>
                                </div>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
    </div>
</div>

<script>
function updateStatus(orderId, status) {
    if (!confirm("Изменить статус заказа #" + orderId + " на \"" + status + "\"?")) {
        return;
    }
    
    const csrfToken = document.querySelector('meta[name="csrf-token"]')?.getAttribute('content') || '<?= $_SESSION['csrf_token'] ?? '' ?>';
    
    fetch('/ajax/admin/update_status.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'X-CSRF-Token': csrfToken
        },
        body: JSON.stringify({
            id: orderId,
            status: status,
            csrf: csrfToken
        })
    })
    .then(res => {
        if (!res.ok) {
            throw new Error('Network response was not ok: ' + res.status);
        }
        return res.json();
    })
    .then(data => {
        if (data.success) {
            alert('Статус заказа успешно обновлён');

            const row = document.querySelector(`tr td:first-child:contains("${orderId}")`)?.closest('tr');
            if (row) {
                const statusCell = row.querySelector('td:nth-child(6)');
                if (statusCell) {
                    statusCell.innerHTML = `
                        <select class="form-select form-select-sm" 
                                onchange="updateStatus(${orderId}, this.value)"
                                style="width: 150px;">
                            ${data.status_options || ''}
                        </select>
                    `;
                }
            }
        } else {
            alert('Ошибка: ' + (data.error || 'Неизвестная ошибка'));
        }
    })
    .catch(error => {
        console.error('Error:', error);
        alert('Ошибка соединения с сервером');
    });
}

function deleteOrder(orderId) {
    if (!confirm("Вы уверены, что хотите удалить заказ #" + orderId + "?")) {
        return;
    }
    
    const csrfToken = document.querySelector('meta[name="csrf-token"]')?.getAttribute('content') || '<?= $_SESSION['csrf_token'] ?? '' ?>';
    
    fetch('/ajax/admin/delete_order.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'X-CSRF-Token': csrfToken
        },
        body: JSON.stringify({
            id: orderId,
            csrf: csrfToken
        })
    })
    .then(res => res.json())
    .then(data => {
        if (data.success) {
            alert('Заказ удалён');
            location.reload();
        } else {
            alert('Ошибка: ' + (data.error || 'Неизвестная ошибка'));
        }
    })
    .catch(error => {
        console.error('Error:', error);
        alert('Ошибка соединения с сервером');
    });
}
</script>

<?php 
require_once __DIR__ . '/../includes/footer.php'; 
?>