<?php

require_once __DIR__ . '/../config/bootstrap.php';
require_once __DIR__ . '/../config/auth.php';
require_once __DIR__ . '/../config/functions.php';

header('Content-Type: application/json');

if (!isset($_SESSION['user_id'])) {
    echo json_encode(['success' => false, 'error' => 'Требуется авторизация']);
    exit;
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode(['success' => false, 'error' => 'Неверный метод запроса']);
    exit;
}

$csrf_token = $_POST['csrf'] ?? $_POST['csrf_token'] ?? '';
if (empty($csrf_token)) {
    echo json_encode(['success' => false, 'error' => 'Отсутствует CSRF токен']);
    exit;
}

if (!check_csrf($csrf_token)) {
    echo json_encode(['success' => false, 'error' => 'Неверный CSRF токен']);
    exit;
}

$user_id = $_SESSION['user_id'];
$email = trim($_POST['email'] ?? '');
$name = trim($_POST['name'] ?? '');
$phone = trim($_POST['phone'] ?? '');
$password = $_POST['password'] ?? '';
$password_confirm = $_POST['password_confirm'] ?? '';
$notifications = (int)($_POST['notifications'] ?? 1);

if (empty($name)) {
    echo json_encode(['success' => false, 'error' => 'Поле "ФИО" обязательно для заполнения']);
    exit;
}

if (empty($email)) {
    echo json_encode(['success' => false, 'error' => 'Поле "Email" обязательно для заполнения']);
    exit;
}

if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
    echo json_encode(['success' => false, 'error' => 'Неверный формат email']);
    exit;
}

if (!empty($phone)) {
    $phone_clean = preg_replace('/\D/', '', $phone);

    if (strlen($phone_clean) < 10) {
        echo json_encode(['success' => false, 'error' => 'Номер телефона слишком короткий. Пример: +7 999 123-45-67']);
        exit;
    }

    if (strlen($phone_clean) === 11 && (strpos($phone_clean, '7') === 0 || strpos($phone_clean, '8') === 0)) {
        $phone = '+7' . substr($phone_clean, 1);
    } elseif (strlen($phone_clean) === 10) {
        $phone = '+7' . $phone_clean;
    } else {
        $phone = '+' . $phone_clean;
    }

    if (!preg_match('/^\+7\d{10}$/', $phone)) {
        echo json_encode(['success' => false, 'error' => 'Неверный формат телефона. Пример: +7 999 123-45-67']);
        exit;
    }
} else {
    $phone = null;
}

$stmt = $pdo->prepare("SELECT id FROM users WHERE email = ? AND id != ?");
$stmt->execute([$email, $user_id]);
if ($stmt->fetch()) {
    echo json_encode(['success' => false, 'error' => 'Email уже используется другим пользователем']);
    exit;
}

if (!empty($password)) {
    if (strlen($password) < 6) {
        echo json_encode(['success' => false, 'error' => 'Пароль должен быть не менее 6 символов']);
        exit;
    }
    
    if ($password !== $password_confirm) {
        echo json_encode(['success' => false, 'error' => 'Пароли не совпадают']);
        exit;
    }
}

try {

    $updateFields = [
        'email = :email',
        'name = :name',
        'phone = :phone',
        'notifications = :notifications'
    ];
    
    $params = [
        'email' => $email,
        'name' => $name,
        'phone' => $phone,
        'notifications' => $notifications,
        'id' => $user_id
    ];

    if (!empty($password)) {
        $updateFields[] = 'password = :password';
        $params['password'] = password_hash($password, PASSWORD_DEFAULT);
    }

    $sql = "UPDATE users SET " . implode(', ', $updateFields) . " WHERE id = :id";
    $stmt = $pdo->prepare($sql);
    $success = $stmt->execute($params);
    
    if ($success) {

        $_SESSION['user_email'] = $email;
        $_SESSION['user_name'] = $name;
        $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
        
        echo json_encode([
            'success' => true,
            'csrf_token' => $_SESSION['csrf_token']
        ]);
    } else {
        echo json_encode(['success' => false, 'error' => 'Ошибка при обновлении данных']);
    }
} catch (Exception $e) {
    error_log("Ошибка обновления настроек: " . $e->getMessage());
    echo json_encode(['success' => false, 'error' => 'Внутренняя ошибка сервера: ' . $e->getMessage()]);
}